<?php
namespace SmartSafeUpdater;

/**
 * Admin UI class.
 *
 * Registers menu pages, enqueues assets and renders the settings screens. It
 * delegates license activation and validation to the License_Manager and
 * conflict scanning to the Conflict_Scanner. Pro features are gated
 * based on the current license status.
 */
class Admin {

    /**
     * License manager instance.
     *
     * @var License_Manager
     */
    protected $license_manager;

    /**
     * Conflict scanner instance.
     *
     * @var Conflict_Scanner
     */
    protected $scanner;

    /**
     * Constructor. Hooks into WordPress admin actions.
     *
     * @param License_Manager   $license_manager License manager instance.
     * @param Conflict_Scanner $scanner         Conflict scanner instance.
     */
    public function __construct( License_Manager $license_manager, Conflict_Scanner $scanner ) {
        $this->license_manager = $license_manager;
        $this->scanner         = $scanner;
        add_action( 'admin_menu', [ $this, 'add_admin_menu' ] );
        add_action( 'admin_post_ssu_activate_license', [ $this, 'handle_activate_license' ] );
        add_action( 'admin_post_ssu_validate_plugin', [ $this, 'handle_validate_plugin' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_assets' ] );
    }

    /**
     * Adds the Smart Safe Updater top-level menu and submenus.
     */
    public function add_admin_menu(): void {
        $capability = 'manage_options';
        $slug       = 'smart-safe-updater';
        // Top-level page
        add_menu_page(
            __( 'Smart Safe Updater', 'smart-safe-updater' ),
            __( 'Smart Safe Updater', 'smart-safe-updater' ),
            $capability,
            $slug,
            [ $this, 'render_overview_page' ],
            'dashicons-shield-alt'
        );
        // License subpage
        add_submenu_page(
            $slug,
            __( 'License', 'smart-safe-updater' ),
            __( 'License', 'smart-safe-updater' ),
            $capability,
            $slug . '-license',
            [ $this, 'render_license_page' ]
        );
        // Scanner subpage
        add_submenu_page(
            $slug,
            __( 'Scanner', 'smart-safe-updater' ),
            __( 'Scanner', 'smart-safe-updater' ),
            $capability,
            $slug . '-scanner',
            [ $this, 'render_scanner_page' ]
        );
    }

    /**
     * Enqueues our admin CSS for a modern look.
     */
    public function enqueue_assets( string $hook_suffix ): void {
        if ( false === strpos( $hook_suffix, 'smart-safe-updater' ) ) {
            return;
        }
        $css_url = plugins_url( 'assets/admin.css', SSU_PLUGIN_FILE );
        wp_enqueue_style( 'ssu-admin-style', $css_url, [], SSU_VERSION );
    }

    /**
     * Renders the overview page. Displays the current license status and a
     * summary of available actions.
     */
    public function render_overview_page(): void {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $status       = $this->license_manager->get_license_status();
        $server_status = $this->license_manager->get_last_server_status();
        $server_msg    = $this->license_manager->get_last_message();
        $status_label = $this->format_status_label( $status );
        ?>
        <div class="wrap ssu-wrap">
            <h1><?php esc_html_e( 'Smart Safe Updater', 'smart-safe-updater' ); ?></h1>

            <?php if ( 'valid' !== $status && in_array( $server_status, [ 'expired', 'inactive' ], true ) ) : ?>
                <div class="notice notice-warning">
                    <p>
                        <?php
                        $fallback = __( 'Your Pro subscription appears to be inactive or expired. Renew to re-enable Pro features.', 'smart-safe-updater' );
                        echo esc_html( $server_msg ? $server_msg : $fallback );
                        ?>
                        <a href="<?php echo esc_url( SSU_RENEW_URL ); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Renew Pro', 'smart-safe-updater' ); ?></a>
                    </p>
                </div>
            <?php endif; ?>
            <p><?php esc_html_e( 'Smart Safe Updater & Conflict Detector helps prevent plugin disasters by scanning for conflicts and testing updates in a sandbox.', 'smart-safe-updater' ); ?></p>

            <table class="widefat ssu-status-table" style="max-width:600px;">
                <thead><tr><th><?php esc_html_e( 'Item', 'smart-safe-updater' ); ?></th><th><?php esc_html_e( 'Status', 'smart-safe-updater' ); ?></th></tr></thead>
                <tbody>
                <tr>
                    <td><?php esc_html_e( 'License', 'smart-safe-updater' ); ?></td>
                    <td><?php echo wp_kses_post( $status_label ); ?></td>
                </tr>
                <tr>
                    <td><?php esc_html_e( 'Conflict Scanner', 'smart-safe-updater' ); ?></td>
                    <td>
                        <?php
                        if ( 'valid' === $status ) {
                            esc_html_e( 'Pro scanner enabled', 'smart-safe-updater' );
                        } else {
                            esc_html_e( 'Basic scan enabled', 'smart-safe-updater' );
                        }
                        ?>
                    </td>
                </tr>
                <tr>
                    <td><?php esc_html_e( 'Safe Update Tester', 'smart-safe-updater' ); ?></td>
                    <td>
                        <?php
                        if ( 'valid' === $status ) {
                            esc_html_e( 'Enabled', 'smart-safe-updater' );
                        } else {
                            printf( '<a href="%s">%s</a>', esc_url( admin_url( 'admin.php?page=smart-safe-updater-license' ) ), esc_html__( 'Upgrade to Pro', 'smart-safe-updater' ) );
                        }
                        ?>
                    </td>
                </tr>
                </tbody>
            </table>
            <p><?php
            printf(
                /* translators: %s: link to license page */
                __( 'Manage your license <a href="%s">here</a>.', 'smart-safe-updater' ),
                esc_url( admin_url( 'admin.php?page=smart-safe-updater-license' ) )
            );
            ?></p>
        </div>
        <?php
    }

    /**
     * Renders the license activation page. Provides a form for entering
     * the license key and displays the current activation status.
     */
    public function render_license_page(): void {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $status   = $this->license_manager->get_license_status();
        $key      = $this->license_manager->get_license_key();
        $server_status = $this->license_manager->get_last_server_status();
        $server_msg    = $this->license_manager->get_last_message();
        $msg      = '';
        if ( ! empty( $_GET['ssu_msg'] ) ) {
            $msg = sanitize_text_field( wp_unslash( $_GET['ssu_msg'] ) );
        }
        ?>
        <div class="wrap ssu-wrap">
            <h1><?php esc_html_e( 'License Management', 'smart-safe-updater' ); ?></h1>

            <?php if ( 'valid' !== $status && in_array( $server_status, [ 'expired', 'inactive' ], true ) ) : ?>
                <div class="notice notice-warning">
                    <p>
                        <?php
                        $fallback = __( 'Your Pro subscription is not active on this site. Renew to re-enable Pro features.', 'smart-safe-updater' );
                        echo esc_html( $server_msg ? $server_msg : $fallback );
                        ?>
                        <a class="button button-primary" href="<?php echo esc_url( SSU_RENEW_URL ); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Renew Pro', 'smart-safe-updater' ); ?></a>
                    </p>
                </div>
            <?php endif; ?>
            <?php if ( $msg ) : ?>
                <div class="notice notice-info is-dismissible"><p><?php echo esc_html( $msg ); ?></p></div>
            <?php endif; ?>
            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <input type="hidden" name="action" value="ssu_activate_license" />
                <?php wp_nonce_field( 'ssu_activate_license' ); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="ssu_license_key"><?php esc_html_e( 'License Key', 'smart-safe-updater' ); ?></label></th>
                        <td><input type="text" id="ssu_license_key" name="ssu_license_key" value="<?php echo esc_attr( $key ); ?>" class="regular-text" /></td>
                    </tr>
                </table>
                <p class="submit"><button type="submit" class="button button-primary"><?php esc_html_e( 'Activate', 'smart-safe-updater' ); ?></button></p>
            </form>
            <p><?php printf( __( 'Current license status: %s', 'smart-safe-updater' ), wp_kses_post( $this->format_status_label( $status ) ) ); ?></p>
        </div>
        <?php
    }

    /**
     * Renders the scanner page. Shows conflict scan results and pro
     * validation controls when the license is valid.
     */
    public function render_scanner_page(): void {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $status  = $this->license_manager->get_license_status();
        $issues  = $this->scanner->scan_conflicts();
        $plugins = [];
        if ( 'valid' === $status ) {
            $plugins = $this->scanner->get_installed_plugins();
        }
        $msg = '';
        if ( ! empty( $_GET['ssu_scan_msg'] ) ) {
            $msg = sanitize_textarea_field( wp_unslash( $_GET['ssu_scan_msg'] ) );
        }
        ?>
        <div class="wrap ssu-wrap">
            <h1><?php esc_html_e( 'Conflict Scanner', 'smart-safe-updater' ); ?></h1>
            <?php if ( $msg ) : ?>
                <div class="notice notice-info is-dismissible"><p><?php echo esc_html( $msg ); ?></p></div>
            <?php endif; ?>
            <h2><?php esc_html_e( 'Detected Conflicts', 'smart-safe-updater' ); ?></h2>
            <?php if ( empty( $issues ) ) : ?>
                <p><?php esc_html_e( 'No known conflicts detected among your active plugins.', 'smart-safe-updater' ); ?></p>
            <?php else : ?>
                <table class="widefat" style="max-width:800px;">
                    <thead><tr><th><?php esc_html_e( 'Plugin', 'smart-safe-updater' ); ?></th><th><?php esc_html_e( 'Conflicts With', 'smart-safe-updater' ); ?></th><th><?php esc_html_e( 'Details', 'smart-safe-updater' ); ?></th></tr></thead>
                    <tbody>
                    <?php foreach ( $issues as $issue ) : ?>
                        <tr>
                            <td><?php echo esc_html( $issue['plugin'] ); ?></td>
                            <td><?php echo esc_html( implode( ', ', (array) $issue['conflicts_with'] ) ); ?></td>
                            <td><?php echo esc_html( $issue['message'] ); ?></td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
            <hr />
            <?php if ( 'valid' === $status ) : ?>
                <h2><?php esc_html_e( 'Pro Plugin Validation', 'smart-safe-updater' ); ?></h2>
                <p><?php esc_html_e( 'Validate plugins for syntax errors before activating or updating them. Select a plugin from the list below and click “Validate”.', 'smart-safe-updater' ); ?></p>
                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                    <input type="hidden" name="action" value="ssu_validate_plugin" />
                    <?php wp_nonce_field( 'ssu_validate_plugin' ); ?>
                    <select name="plugin_file">
                        <?php foreach ( $plugins as $file => $data ) : ?>
                            <option value="<?php echo esc_attr( $file ); ?>"><?php echo esc_html( $data['Name'] . ' (' . $file . ')' ); ?></option>
                        <?php endforeach; ?>
                    </select>
                    <button type="submit" class="button button-secondary"><?php esc_html_e( 'Validate', 'smart-safe-updater' ); ?></button>
                </form>
            <?php else : ?>
                <h2><?php esc_html_e( 'Pro Plugin Validation', 'smart-safe-updater' ); ?></h2>
                <p><?php esc_html_e( 'Upgrade to Pro to validate plugins for syntax errors before activating them.', 'smart-safe-updater' ); ?></p>
                <p><a class="button button-primary" href="<?php echo esc_url( admin_url( 'admin.php?page=smart-safe-updater-license' ) ); ?>"><?php esc_html_e( 'Upgrade to Pro', 'smart-safe-updater' ); ?></a></p>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Handles the license activation form submission.
     */
    public function handle_activate_license(): void {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( __( 'You do not have permission to perform this action.', 'smart-safe-updater' ) );
        }
        check_admin_referer( 'ssu_activate_license' );
        $key    = isset( $_POST['ssu_license_key'] ) ? sanitize_text_field( wp_unslash( $_POST['ssu_license_key'] ) ) : '';
        $result = $this->license_manager->activate_license( $key );
        $msg    = $result['msg'] ?? '';
        $redirect = add_query_arg( 'ssu_msg', rawurlencode( $msg ), admin_url( 'admin.php?page=smart-safe-updater-license' ) );
        wp_safe_redirect( $redirect );
        exit;
    }

    /**
     * Handles plugin validation requests in the Pro version.
     */
    public function handle_validate_plugin(): void {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( __( 'You do not have permission to perform this action.', 'smart-safe-updater' ) );
        }
        check_admin_referer( 'ssu_validate_plugin' );
        if ( ! $this->license_manager->is_license_valid() ) {
            wp_die( __( 'Invalid license. Please activate your license to use this feature.', 'smart-safe-updater' ) );
        }
        $plugin_file = isset( $_POST['plugin_file'] ) ? sanitize_text_field( wp_unslash( $_POST['plugin_file'] ) ) : '';
        if ( '' === $plugin_file ) {
            $msg = __( 'No plugin selected.', 'smart-safe-updater' );
        } else {
            $result = $this->scanner->validate_plugin_file( $plugin_file );
            $msg    = $result['msg'];
        }
        $redirect = add_query_arg( 'ssu_scan_msg', rawurlencode( $msg ), admin_url( 'admin.php?page=smart-safe-updater-scanner' ) );
        wp_safe_redirect( $redirect );
        exit;
    }

    /**
     * Formats a license status into a coloured badge.
     *
     * @param string $status Internal status string.
     * @return string HTML span element with classes.
     */
    protected function format_status_label( string $status ): string {
        switch ( $status ) {
            case 'valid':
                $class = 'ssu-badge ssu-badge-success';
                $text  = __( 'Valid', 'smart-safe-updater' );
                break;
            case 'invalid':
                $class = 'ssu-badge ssu-badge-error';
                $text  = __( 'Invalid', 'smart-safe-updater' );
                break;
            case 'inactive':
                $class = 'ssu-badge ssu-badge-warning';
                $text  = __( 'Inactive', 'smart-safe-updater' );
                break;
            case 'site_limit':
                $class = 'ssu-badge ssu-badge-warning';
                $text  = __( 'Site limit reached', 'smart-safe-updater' );
                break;
            default:
                $class = 'ssu-badge ssu-badge-neutral';
                $text  = __( 'Not activated', 'smart-safe-updater' );
                break;
        }
        return '<span class="' . esc_attr( $class ) . '">' . esc_html( $text ) . '</span>';
    }
}