<?php
namespace SmartSafeUpdater;

use WP_Error;

/**
 * Conflict and error scanner.
 *
 * Scans active plugins for known conflicts and performs lightweight
 * validation on plugin code. When used in the Pro version, this class can
 * validate plugin files for syntax errors using WordPress core functions.
 */
class Conflict_Scanner {

    /**
     * Known conflict pairs. Each entry maps a plugin basename to an array
     * of basenames that conflict with it. This list is illustrative; real
     * conflicts should be collected from support reports or public
     * vulnerability disclosures.
     *
     * @var array<string,array<string>>
     */
    protected $known_conflicts = [
        // Example: Hello Dolly conflicts with itself and Akismet (fictional)
        'hello-dolly/hello.php' => [
            'akismet/akismet.php',
        ],
    ];

    /**
     * Performs a simple scan of the active plugins to detect known conflicts.
     * Returns an array of associative arrays describing each issue.
     *
     * Each issue contains:
     *  - 'plugin': basename of the plugin
     *  - 'conflicts_with': array of conflicting plugin basenames
     *  - 'message': human-readable description
     *
     * @return array<int,array<string,mixed>>
     */
    public function scan_conflicts(): array {
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $active_plugins = (array) get_option( 'active_plugins', [] );
        $issues         = [];
        foreach ( $active_plugins as $plugin ) {
            if ( isset( $this->known_conflicts[ $plugin ] ) ) {
                $conflicts = [];
                foreach ( $this->known_conflicts[ $plugin ] as $conflict ) {
                    if ( in_array( $conflict, $active_plugins, true ) ) {
                        $conflicts[] = $conflict;
                    }
                }
                if ( ! empty( $conflicts ) ) {
                    $issues[] = [
                        'plugin'        => $plugin,
                        'conflicts_with' => $conflicts,
                        'message'       => sprintf(
                            /* translators: 1: plugin slug, 2: comma-separated list of conflicting plugins */
                            __( '%1$s conflicts with %2$s.', 'smart-safe-updater' ),
                            esc_html( $plugin ),
                            esc_html( implode( ', ', $conflicts ) )
                        ),
                    ];
                }
            }
        }
        return $issues;
    }

    /**
     * Validates the PHP file of a plugin for syntax errors. This is a
     * lightweight check and does not execute the plugin. It uses
     * WordPress's validate_plugin() function, which attempts to include
     * the file in a sandbox and catches parse errors. Note that this does
     * not detect runtime issues.
     *
     * @param string $plugin_file Plugin basename (relative to plugins directory).
     * @return array{ok:bool,msg:string}
     */
    public function validate_plugin_file( string $plugin_file ): array {
        if ( ! function_exists( 'validate_plugin' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $error = validate_plugin( $plugin_file );
        if ( is_wp_error( $error ) ) {
            $msg = '';
            foreach ( $error->get_error_messages() as $m ) {
                $msg .= $m . "\n";
            }
            return [ 'ok' => false, 'msg' => trim( $msg ) ];
        }
        return [ 'ok' => true, 'msg' => __( 'Plugin passed syntax check.', 'smart-safe-updater' ) ];
    }

    /**
     * Returns an array of installed plugins using WordPress core. The array
     * keys are the plugin basenames (e.g. `hello-dolly/hello.php`) and the
     * values contain metadata including name and version. Requires
     * admin_includes/plugin.php. This wrapper is provided for convenience.
     *
     * @return array<string,array<string,mixed>>
     */
    public function get_installed_plugins(): array {
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        return get_plugins();
    }
}