<?php
namespace SmartSafeUpdater;

use WP_Error;

/**
 * Handles activation and validation of the Pro license.
 *
 * The license manager communicates with the Sparkcut Labs license server via
 * HTTP POST requests. It stores the license key and status in WordPress
 * options and exposes helper methods to check whether the current site
 * holds a valid license. A valid license unlocks all Pro features.
 */
class License_Manager {

    /**
     * Option name for storing the license key.
     *
     * @var string
     */
    protected $key_option = 'ssu_license_key';

    /**
     * Option name for storing the license status.
     *
     * Possible values: `valid`, `invalid`, `inactive`, `site_limit`, `missing`.
     *
     * @var string
     */
    protected $status_option = 'ssu_license_status';

    /**
     * Option name for storing the timestamp of the last validation.
     *
     * @var string
     */
    protected $checked_option = 'ssu_license_checked';

    /**
     * Option name for storing the last message from the license server.
     *
     * @var string
     */
    protected $last_msg_option = 'ssu_license_last_msg';

    /**
     * Option name for storing the last status code returned by the license server.
     * (e.g. valid, inactive, expired, domain_mismatch)
     *
     * @var string
     */
    protected $last_status_option = 'ssu_license_last_status';

    /**
     * Cached license key.
     *
     * @var string
     */
    protected $key = '';

    /**
     * Cached license status.
     *
     * @var string
     */
    protected $status = '';

    /**
     * Cached last server message.
     *
     * @var string
     */
    protected $last_msg = '';

    /**
     * Cached last server status.
     *
     * @var string
     */
    protected $last_server_status = '';

    /**
     * Constructor. Loads the current license details.
     */
    public function __construct() {
        $this->key    = (string) get_option( $this->key_option, '' );
        $this->status = (string) get_option( $this->status_option, 'missing' );
        $this->last_msg = (string) get_option( $this->last_msg_option, '' );
        $this->last_server_status = (string) get_option( $this->last_status_option, '' );

        // Hook to validate the license periodically when visiting admin.
        add_action( 'admin_init', [ $this, 'maybe_validate_license' ] );
    }

    /**
     * Returns the last message returned by the license server.
     */
    public function get_last_message(): string {
        return (string) $this->last_msg;
    }

    /**
     * Returns the last server status code (if the server provided one).
     */
    public function get_last_server_status(): string {
        return (string) $this->last_server_status;
    }

    /**
     * Returns the stored license key.
     *
     * @return string
     */
    public function get_license_key(): string {
        return (string) $this->key;
    }

    /**
     * Returns the current license status.
     *
     * @return string One of `valid`, `invalid`, `inactive`, `site_limit`, `missing`.
     */
    public function get_license_status(): string {
        return $this->status ?: 'missing';
    }

    /**
     * Determines whether the current license is valid.
     *
     * @return bool
     */
    public function is_license_valid(): bool {
        return 'valid' === $this->get_license_status();
    }

    /**
     * Activates a license for this site. Attempts to register the site URL
     * with the license server so that future validations succeed. On success,
     * the license status is set to `valid` and the key is stored. On
     * failure, the status is set to the returned error state.
     *
     * @param string $key User-provided license key.
     * @return array{ok:bool,msg:string,status:string}
     */
    public function activate_license( string $key ): array {
        $key = trim( $key );
        if ( '' === $key ) {
            $this->status = 'missing';
            update_option( $this->status_option, $this->status );
            update_option( $this->key_option, '' );
            return [ 'ok' => false, 'msg' => __( 'Please enter a license key.', 'smart-safe-updater' ), 'status' => $this->status ];
        }

        $domain = $this->get_current_domain();
        $post   = [
            'action'      => 'activate',
            'license_key' => $key,
            'product'     => SSU_PRODUCT_CODE,
            'domain'      => $domain,
        ];
        $response = $this->post_to_server( $post );
        if ( is_wp_error( $response ) ) {
            return [ 'ok' => false, 'msg' => $response->get_error_message(), 'status' => 'invalid' ];
        }
        $data = json_decode( wp_remote_retrieve_body( $response ), true );
        if ( ! is_array( $data ) || empty( $data['ok'] ) ) {
            $msg = $data['msg'] ?? __( 'Unexpected response from license server.', 'smart-safe-updater' );
            $this->status = 'invalid';
            $this->last_msg = (string) $msg;
            $this->last_server_status = (string) ( $data['status'] ?? '' );
            update_option( $this->status_option, $this->status );
            update_option( $this->last_msg_option, $this->last_msg );
            update_option( $this->last_status_option, $this->last_server_status );
            return [ 'ok' => false, 'msg' => (string) $msg, 'status' => $this->status ];
        }

        // Success. Store key and status.
        $this->key    = $key;
        $this->status = 'valid';
        $this->last_msg = '';
        $this->last_server_status = 'valid';
        update_option( $this->key_option, $this->key );
        update_option( $this->status_option, $this->status );
        update_option( $this->last_msg_option, $this->last_msg );
        update_option( $this->last_status_option, $this->last_server_status );
        update_option( $this->checked_option, time() );

        return [ 'ok' => true, 'msg' => __( 'License activated!', 'smart-safe-updater' ), 'status' => $this->status ];
    }

    /**
     * Validates the current license. This should be called periodically to
     * ensure the stored key remains active and associated with this domain.
     * It updates the internal status based on the API response.
     *
     * @return array{ok:bool,msg:string,status:string}
     */
    public function validate_license(): array {
        $key = $this->get_license_key();
        if ( '' === $key ) {
            $this->status = 'missing';
            update_option( $this->status_option, $this->status );
            return [ 'ok' => false, 'msg' => __( 'No license key saved.', 'smart-safe-updater' ), 'status' => $this->status ];
        }
        $domain = $this->get_current_domain();
        $post   = [
            'action'      => 'validate',
            'license_key' => $key,
            'product'     => SSU_PRODUCT_CODE,
            'domain'      => $domain,
        ];
        $response = $this->post_to_server( $post );
        if ( is_wp_error( $response ) ) {
            return [ 'ok' => false, 'msg' => $response->get_error_message(), 'status' => 'invalid' ];
        }
        $data = json_decode( wp_remote_retrieve_body( $response ), true );
        if ( ! is_array( $data ) || empty( $data['ok'] ) ) {
            $server_status = (string) ( $data['status'] ?? '' );
            // Map server status -> internal status, so Admin UI can show a clearer badge.
            switch ( $server_status ) {
                case 'inactive':
                    $this->status = 'inactive';
                    break;
                case 'expired':
                    $this->status = 'inactive'; // treat expired as inactive in gating logic
                    break;
                case 'site_limit':
                    $this->status = 'site_limit';
                    break;
                case 'domain_mismatch':
                default:
                    $this->status = 'invalid';
                    break;
            }
            update_option( $this->status_option, $this->status );
            $msg = $data['msg'] ?? __( 'License validation failed.', 'smart-safe-updater' );
            $this->last_msg = (string) $msg;
            $this->last_server_status = $server_status;
            update_option( $this->last_msg_option, $this->last_msg );
            update_option( $this->last_status_option, $this->last_server_status );
            return [ 'ok' => false, 'msg' => (string) $msg, 'status' => $this->status ];
        }
        $this->status = 'valid';
        $this->last_msg = '';
        $this->last_server_status = (string) ( $data['status'] ?? 'valid' );
        update_option( $this->status_option, $this->status );
        update_option( $this->last_msg_option, $this->last_msg );
        update_option( $this->last_status_option, $this->last_server_status );
        update_option( $this->checked_option, time() );
        return [ 'ok' => true, 'msg' => __( 'License valid.', 'smart-safe-updater' ), 'status' => $this->status ];
    }

    /**
     * Determines whether the license should be validated. If more than 24 hours
     * have passed since the last check, it runs validation in the background.
     */
    public function maybe_validate_license(): void {
        // Only run in admin context and avoid running during AJAX/cron.
        if ( ! is_admin() || defined( 'DOING_AJAX' ) || defined( 'DOING_CRON' ) ) {
            return;
        }
        $last_checked = (int) get_option( $this->checked_option, 0 );
        // 24 hours = 24 * 60 * 60
        if ( $last_checked > 0 && ( time() - $last_checked ) < DAY_IN_SECONDS ) {
            return;
        }
        // We don't care about the response message here; this call updates
        // options as a side effect.
        $this->validate_license();
    }

    /**
     * Makes a POST request to the license server. Uses WordPress HTTP API
     * functions so it is compatible with most hosting environments. Adds a
     * timeout and sets a custom user-agent for easier debugging on the server.
     *
     * @param array $body Data to send via POST.
     * @return array|WP_Error
     */
    protected function post_to_server( array $body ) {
        $args = [
            'body'        => $body,
            'timeout'     => 15,
            'user-agent'  => 'SmartSafeUpdater/' . SSU_VERSION . '; ' . home_url(),
            'redirection' => 3,
        ];
        // Make the request.
        $response = wp_remote_post( SSU_LICENSE_SERVER, $args );
        if ( is_wp_error( $response ) ) {
            return $response;
        }
        // A non-200 response is treated as an error.
        $code = (int) wp_remote_retrieve_response_code( $response );
        if ( $code < 200 || $code >= 300 ) {
            return new WP_Error( 'ssu_http_error', sprintf( __( 'License server returned HTTP %s.', 'smart-safe-updater' ), $code ) );
        }
        return $response;
    }

    /**
     * Determines the current domain for activation/validation. Uses the
     * `home_url` and strips common prefixes. This matches the domain
     * normalization logic used by the license server.
     *
     * @return string Domain without protocol or path.
     */
    protected function get_current_domain(): string {
        $url = home_url();
        // Remove protocol
        $url = preg_replace( '#^https?://#i', '', $url );
        // Remove trailing slash and path
        $url = trim( strtok( $url, '/' ) );
        return strtolower( $url );
    }
}