<?php
/**
 * Smart Safe Updater & Conflict Detector
 *
 * This plugin provides a simple way to test plugin updates safely and scan
 * for common conflicts before they take your site down. The free version
 * performs a lightweight scan of installed plugins, while the Pro version
 * unlocks sandboxed validation and scheduled safety checks. Licenses are
 * managed through the existing Sparkcut Labs license server and Stripe
 * checkout workflow. See the plugin settings page for activation.
 *
 * Plugin Name: Smart Safe Updater & Conflict Detector
 * Plugin URI:  https://sparkcutlabs.com/
 * Description: Safely test plugin updates and detect conflicts. Activate a Pro license for sandboxed testing and advanced checks.
 * Version:     1.0.0
 * Author:      Sparkcut Labs
 * Author URI:  https://sparkcutlabs.com/
 * Text Domain: smart-safe-updater
 * Domain Path: /languages
 * License:     GPLv2 or later
 *
 * @package SmartSafeUpdater
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// -----------------------------------------------------------------------------
// Constants
//
// These values define the product code and license server endpoint. If you
// change these values, ensure the corresponding product exists on your
// license server and is configured in your Stripe checkout/webhook handler.

/**
 * Current plugin version.
 */
if ( ! defined( 'SSU_VERSION' ) ) {
    define( 'SSU_VERSION', '1.0.0' );
}

/**
 * Main file path.
 */
if ( ! defined( 'SSU_PLUGIN_FILE' ) ) {
    define( 'SSU_PLUGIN_FILE', __FILE__ );
}

/**
 * Directory path to the plugin.
 */
if ( ! defined( 'SSU_PLUGIN_DIR' ) ) {
    define( 'SSU_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
}

/**
 * License server API endpoint. You can override this constant in wp-config.php
 * if you run your own license server. The default points to the Sparkcut
 * license API.
 */
if ( ! defined( 'SSU_LICENSE_SERVER' ) ) {
    define( 'SSU_LICENSE_SERVER', 'https://sparkcutlabs.com/license-server/api.php' );
}

/**
 * Unique product identifier used by the license server. This value must
 * correspond to the product configured in your checkout and webhook code.
 */
if ( ! defined( 'SSU_PRODUCT_CODE' ) ) {
    define( 'SSU_PRODUCT_CODE', 'ssucp' );
}

/**
 * Public pricing/renewal URL shown when a license is inactive/expired.
 * Update this to the correct landing page on your site.
 */
if ( ! defined( 'SSU_RENEW_URL' ) ) {
    define( 'SSU_RENEW_URL', 'https://sparkcutlabs.com/index.html#pricing-ssu' );
}

// -----------------------------------------------------------------------------
// Autoloader
//
// Basic PSR-4 autoloader for our classes. Namespaces begin with
// `SmartSafeUpdater\` and map to the `includes/` directory.

spl_autoload_register( function ( $class ) {
    if ( 0 !== strpos( $class, 'SmartSafeUpdater\' ) ) {
        return;
    }
    $path = SSU_PLUGIN_DIR . 'includes/' . str_replace( '\\', '/', substr( $class, strlen( 'SmartSafeUpdater\\' ) ) ) . '.php';
    if ( file_exists( $path ) ) {
        require_once $path;
    }
} );

// -----------------------------------------------------------------------------
// Kick off the plugin
//
// Instantiate the main plugin class when all plugins are loaded. This ensures
// that WordPress functions and other plugins are available.

add_action( 'plugins_loaded', function () {
    // Load translations
    load_plugin_textdomain( 'smart-safe-updater', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );

    // Initialise license manager and admin UI
    $license_manager = new SmartSafeUpdater\License_Manager();
    $conflict_scanner = new SmartSafeUpdater\Conflict_Scanner();
    new SmartSafeUpdater\Admin( $license_manager, $conflict_scanner );
} );