/*
 * Sparkcut AI Chatbot Pro front‑end logic
 * Handles chat panel, suggestions, product cards, checkout mode and AI interaction
 */
(function($){
    const state = {
        open: false,
        messages: [],
        loading: false,
    };
    function el(html) {
        const t = document.createElement('template');
        t.innerHTML = html.trim();
        return t.content.firstChild;
    }
    function escapeHtml(str) {
        return String(str).replaceAll('&','&amp;').replaceAll('<','&lt;').replaceAll('>','&gt;').replaceAll('"','&quot;').replaceAll("'",'&#039;');
    }
    function render() {
        const root = document.getElementById('sacp-root');
        if(!root) return;
        root.innerHTML = '';
        const pos   = root.dataset.position || 'right';
        const theme = root.dataset.theme || 'auto';
        const wrap = el(
            `<div class="sacp-wrap sacp-${pos} sacp-theme-${theme}">
                <button class="sacp-bubble" aria-label="Chat">
                    <span class="sacp-bubble-dot"></span>
                    <span class="sacp-bubble-text">Chat</span>
                </button>
                <div class="sacp-panel" style="display:${state.open?'block':'none'}">
                    <div class="sacp-head">
                        <div class="sacp-title">AI Assistant</div>
                        <button class="sacp-close" aria-label="Close">×</button>
                    </div>
                    <div class="sacp-body" id="sacp-body"></div>
                    <div class="sacp-suggest" id="sacp-suggest"></div>
                    <div class="sacp-foot">
                        <input type="text" id="sacp-input" placeholder="Type a message…" autocomplete="off" />
                        <button id="sacp-send">→</button>
                    </div>
                    <div class="sacp-note" id="sacp-note"></div>
                </div>
            </div>`
        );
        root.appendChild(wrap);
        const bubble = wrap.querySelector('.sacp-bubble');
        const close  = wrap.querySelector('.sacp-close');
        bubble.addEventListener('click', () => {
            state.open = true;
            render();
            bootWelcome();
        });
        close.addEventListener('click', () => {
            state.open = false;
            render();
        });
        const input = wrap.querySelector('#sacp-input');
        const sendBtn = wrap.querySelector('#sacp-send');
        sendBtn.addEventListener('click', () => sendMessage(input.value));
        input.addEventListener('keydown', (e) => {
            if(e.key === 'Enter') {
                sendMessage(input.value);
            }
        });
        repaintMessages();
        repaintSuggestions();
        repaintNote();
    }
    function bootWelcome() {
        if(state.messages.length) return;
        const welcome = SACP.onCheckout ? SACP.settings.checkoutWelcome : SACP.settings.welcome;
        state.messages.push({role:'assistant', content: welcome});
        repaintMessages();
    }
    function repaintMessages() {
        const body = document.getElementById('sacp-body');
        if(!body) return;
        body.innerHTML = '';
        state.messages.forEach(m => {
            const cls = m.role === 'assistant' ? 'sacp-msg-a' : 'sacp-msg-u';
            const bub = el(`<div class="sacp-msg ${cls}"><div class="sacp-bub">${escapeHtml(m.content)}</div></div>`);
            body.appendChild(bub);
        });
        body.scrollTop = body.scrollHeight;
    }
    function repaintSuggestions() {
        const box = document.getElementById('sacp-suggest');
        if(!box) return;
        const items = SACP.onCheckout ? (SACP.settings.checkoutSuggestions || []) : (SACP.settings.suggestions || []);
        box.innerHTML = '';
        items.slice(0,4).forEach(txt => {
            const b = el(`<button class="sacp-chip">${escapeHtml(txt)}</button>`);
            b.addEventListener('click', () => {
                sendMessage(txt);
            });
            box.appendChild(b);
        });
    }
    function repaintNote(msg) {
        const note = document.getElementById('sacp-note');
        if(!note) return;
        if(msg) {
            note.style.display = 'block';
            note.textContent = msg;
        } else {
            if(!SACP.license || !SACP.license.valid) {
                note.style.display = 'block';
                note.textContent = 'Pro license inactive. Activate in Settings → Sparkcut AI Chatbot.';
            } else {
                note.style.display = 'none';
                note.textContent = '';
            }
        }
    }
    async function sendMessage(text) {
        text = (text || '').trim();
        if(!text || state.loading) return;
        state.messages.push({role:'user', content: text});
        repaintMessages();
        const input = document.getElementById('sacp-input');
        if(input) input.value = '';
        // show loading
        state.loading = true;
        state.messages.push({role:'assistant', content:'…'});
        repaintMessages();
        try {
            const res = await $.post(SACP.ajax, {
                action: 'sacp_chat',
                nonce: SACP.nonce,
                user_text: text,
                messages: state.messages.filter(m => m.content !== '…'),
            });
            // remove loading
            state.messages.pop();
            state.loading = false;
            if(!res || !res.success) {
                repaintNote(res?.data?.message || 'Error');
                state.messages.push({role:'assistant', content: res?.data?.message || 'Sorry, something went wrong.'});
                repaintMessages();
                return;
            }
            repaintNote('');
            const reply = res.data.reply || 'Okay.';
            state.messages.push({role:'assistant', content: reply});
            repaintMessages();
            // handle products directive
            if(res.data.products_directive && res.data.products_directive.q) {
                await showProductCards(res.data.products_directive.q, res.data.products_directive.max || 6);
            }
        } catch(e) {
            // remove loading and show error
            state.messages.pop();
            state.loading = false;
            repaintNote('Request failed');
            state.messages.push({role:'assistant', content:'Request failed.'});
            repaintMessages();
        }
    }
    async function showProductCards(q, max) {
        try {
            const res = await $.post(SACP.ajax, {
                action: 'sacp_search_products',
                nonce: SACP.nonce,
                q: q,
                max: max
            });
            if(!res || !res.success) return;
            const prods = res.data.products || [];
            if(!prods.length) return;
            const body = document.getElementById('sacp-body');
            if(!body) return;
            const wrap = el('<div class="sacp-cards"></div>');
            prods.forEach(p => {
                const card = el(
                    `<div class="sacp-card">
                        <a class="sacp-card-img" href="${p.url}" target="_blank" rel="noopener">
                            <img src="${p.image}" alt="" />
                        </a>
                        <div class="sacp-card-meta">
                            <a class="sacp-card-title" href="${p.url}" target="_blank" rel="noopener">${escapeHtml(p.name)}</a>
                            <div class="sacp-card-price">${p.price_html}</div>
                        </div>
                        <button class="sacp-card-btn" ${p.in_stock ? '' : 'disabled'} data-id="${p.id}">${p.in_stock ? 'Add to cart' : 'Out of stock'}</button>
                    </div>`
                );
                const btn = card.querySelector('.sacp-card-btn');
                btn.addEventListener('click', async () => {
                    if(btn.disabled) return;
                    const id = btn.dataset.id;
                    btn.disabled = true;
                    const original = btn.textContent;
                    btn.textContent = 'Adding…';
                    try {
                        const add = await $.post(SACP.ajax, {
                            action: 'sacp_add_to_cart',
                            nonce: SACP.nonce,
                            product_id: id,
                            qty: 1
                        });
                        if(add && add.success) {
                            btn.textContent = 'Added ✓';
                        } else {
                            btn.textContent = 'Failed';
                        }
                    } catch(e) {
                        btn.textContent = 'Failed';
                    }
                    setTimeout(() => {
                        btn.disabled = false;
                        btn.textContent = original;
                    }, 1200);
                });
                wrap.appendChild(card);
            });
            body.appendChild(wrap);
            body.scrollTop = body.scrollHeight;
        } catch(e) {
            // ignore
        }
    }
    $(document).ready(render);
})(jQuery);