<?php
/**
 * Plugin Name: Sparkcut AI Chatbot Pro
 * Description: Shop‑aware AI chatbot with product cards, add‑to‑cart, checkout assistance (shipping cutoff awareness and policy answers) and license validation. Works without AI when needed.
 * Version: 0.2.0
 * Author: Sparkcut Labs
 * Text Domain: sparkcut-ai-chatbot-pro
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

final class Sparkcut_AI_Chatbot_Pro {
    const VERSION         = '0.2.0';
    const OPT_KEY         = 'sacp_settings';
    const TRANS_LICENSE   = 'sacp_license_state';

    private static $instance = null;
    public static function instance() {
        if (!self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        // front
        add_action('init', [$this, 'maybe_register_assets']);
        add_action('wp_footer', [$this, 'render_widget']);

        // admin
        add_action('admin_menu', [$this, 'admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);

        // ajax
        add_action('wp_ajax_sacp_chat', [$this, 'ajax_chat']);
        add_action('wp_ajax_nopriv_sacp_chat', [$this, 'ajax_chat']);
        add_action('wp_ajax_sacp_search_products', [$this, 'ajax_search_products']);
        add_action('wp_ajax_nopriv_sacp_search_products', [$this, 'ajax_search_products']);
        add_action('wp_ajax_sacp_add_to_cart', [$this, 'ajax_add_to_cart']);
        add_action('wp_ajax_nopriv_sacp_add_to_cart', [$this, 'ajax_add_to_cart']);
    }

    /**
     * Default settings
     */
    public function defaults() {
        return [
            'enabled'                => 'yes',
            'position'               => 'right',
            'theme'                  => 'auto',
            'welcome'                => 'Hi! How can I help you today?',
            'checkout_welcome'       => 'I can help with shipping times, returns or your order before checkout.',
            'suggestions'            => "Show me items under €50\nBest sellers\nWhat ships today?",
            'checkout_suggestions'   => "Will this ship today?\nWhat is your returns policy?\nI have a question about checkout",
            'gemini_api_key'         => '',
            'gemini_model'           => 'gemini-1.5-flash',
            'system_prompt'          => 'You are a helpful e‑commerce assistant. Ask brief follow‑up questions when needed. Recommend relevant WooCommerce products.',
            // checkout assistance options
            'checkout_enable'        => 'yes',
            'cutoff_time'            => '16:00', // HH:MM 24h
            'cutoff_days'            => '1,2,3,4,5', // 0=Sun ... 6=Sat
            'cutoff_timezone'        => wp_timezone_string(),
            'dispatch_today_msg'     => 'Orders placed now will ship today.',
            'dispatch_next_msg'      => 'Orders placed now will ship the next working day.',
            'knowledge_shipping'     => 'We offer worldwide shipping. Delivery times vary by destination.',
            'knowledge_returns'      => 'You can return items within 14 days of delivery. Items must be unused.',
            'knowledge_contact'      => 'Reach us at support@example.com or call +1 234 567 890.',
            'license_key'            => '',
            'license_server'         => 'https://sparkcutlabs.com/license-server/api.php',
            'product_id'             => 'sparkcut-ai-chatbot-pro',
        ];
    }

    /**
     * Retrieve merged settings
     */
    public function get_settings() {
        $saved   = get_option(self::OPT_KEY, []);
        $defaults = $this->defaults();
        return wp_parse_args($saved, $defaults);
    }

    /**
     * Register assets on front end when enabled
     */
    public function maybe_register_assets() {
        $s = $this->get_settings();
        if ($s['enabled'] !== 'yes') {
            return;
        }
        $url = plugin_dir_url(__FILE__);
        // register assets
        wp_register_style('sacp-style', $url . 'assets/chat.css', [], self::VERSION);
        wp_register_script('sacp-script', $url . 'assets/chat.js', ['jquery'], self::VERSION, true);
        wp_enqueue_style('sacp-style');
        wp_enqueue_script('sacp-script');
        // Determine if we are on cart/checkout page
        $on_checkout = false;
        if (function_exists('is_cart') && function_exists('is_checkout')) {
            $on_checkout = is_cart() || is_checkout();
        }
        // pass data to JS
        wp_localize_script('sacp-script', 'SACP', [
            'ajax'      => admin_url('admin-ajax.php'),
            'nonce'     => wp_create_nonce('sacp_nonce'),
            'settings'  => [
                'position'  => $s['position'],
                'theme'     => $s['theme'],
                'welcome'   => $s['welcome'],
                'checkoutWelcome' => $s['checkout_welcome'],
                'suggestions'      => $this->text_to_array($s['suggestions']),
                'checkoutSuggestions' => $this->text_to_array($s['checkout_suggestions']),
            ],
            'wc'        => [
                'currency_symbol' => function_exists('get_woocommerce_currency_symbol') ? get_woocommerce_currency_symbol() : '€',
            ],
            'license'   => $this->license_state(),
            'onCheckout' => $on_checkout,
        ]);
    }

    /**
     * Helper: convert textarea lines into array of trimmed non-empty strings
     */
    private function text_to_array($text) {
        $lines = preg_split("/(\r\n|\n|\r)/", (string) $text);
        $out   = [];
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line !== '') {
                $out[] = $line;
            }
        }
        return $out;
    }

    /**
     * Render root container in footer
     */
    public function render_widget() {
        $s = $this->get_settings();
        if ($s['enabled'] !== 'yes') {
            return;
        }
        echo '<div id="sacp-root" data-position="' . esc_attr($s['position']) . '" data-theme="' . esc_attr($s['theme']) . '"></div>';
    }

    /**
     * Admin menu
     */
    public function admin_menu() {
        add_options_page(
            __('Sparkcut AI Chatbot Pro', 'sparkcut-ai-chatbot-pro'),
            __('Sparkcut AI Chatbot Pro', 'sparkcut-ai-chatbot-pro'),
            'manage_options',
            'sparkcut-ai-chatbot-pro',
            [$this, 'admin_page']
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('sacp_settings_group', self::OPT_KEY, [
            'type'              => 'array',
            'sanitize_callback' => [$this, 'sanitize_settings'],
        ]);
    }

    /**
     * Sanitize settings on save
     */
    public function sanitize_settings($input) {
        $defaults = $this->defaults();
        $out = [];
        foreach ($defaults as $k => $v) {
            if (!isset($input[$k])) {
                continue;
            }
            // treat checkboxes
            if ($k === 'enabled' || $k === 'checkout_enable') {
                $out[$k] = ($input[$k] === 'yes') ? 'yes' : 'no';
                continue;
            }
            $out[$k] = is_string($input[$k]) ? sanitize_textarea_field($input[$k]) : $input[$k];
        }
        // clear license cache when license key changed
        delete_transient(self::TRANS_LICENSE);
        return wp_parse_args($out, $defaults);
    }

    /**
     * Admin page HTML
     */
    public function admin_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        $s      = $this->get_settings();
        $license = $this->license_state(true);
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Sparkcut AI Chatbot Pro', 'sparkcut-ai-chatbot-pro'); ?></h1>
            <form method="post" action="options.php">
                <?php settings_fields('sacp_settings_group'); ?>
                <table class="form-table">
                    <tr><th scope="row">Enable</th><td><label><input type="checkbox" name="<?php echo esc_attr(self::OPT_KEY); ?>[enabled]" value="yes" <?php checked($s['enabled'], 'yes'); ?>/> Enable chatbot</label></td></tr>
                    <tr><th scope="row">Position</th><td><select name="<?php echo esc_attr(self::OPT_KEY); ?>[position]"><option value="right" <?php selected($s['position'], 'right'); ?>>Bottom Right</option><option value="left" <?php selected($s['position'], 'left'); ?>>Bottom Left</option></select></td></tr>
                    <tr><th scope="row">Theme</th><td><select name="<?php echo esc_attr(self::OPT_KEY); ?>[theme]"><option value="auto" <?php selected($s['theme'], 'auto'); ?>>Auto</option><option value="light" <?php selected($s['theme'], 'light'); ?>>Light</option><option value="dark" <?php selected($s['theme'], 'dark'); ?>>Dark</option></select></td></tr>
                    <tr><th scope="row">Welcome message</th><td><input type="text" class="regular-text" name="<?php echo esc_attr(self::OPT_KEY); ?>[welcome]" value="<?php echo esc_attr($s['welcome']); ?>" /></td></tr>
                    <tr><th scope="row">Checkout welcome message</th><td><input type="text" class="regular-text" name="<?php echo esc_attr(self::OPT_KEY); ?>[checkout_welcome]" value="<?php echo esc_attr($s['checkout_welcome']); ?>" /></td></tr>
                    <tr><th scope="row">Suggestion buttons</th><td><textarea class="large-text" rows="4" name="<?php echo esc_attr(self::OPT_KEY); ?>[suggestions]"><?php echo esc_textarea($s['suggestions']); ?></textarea><p class="description">One per line</p></td></tr>
                    <tr><th scope="row">Checkout suggestion buttons</th><td><textarea class="large-text" rows="4" name="<?php echo esc_attr(self::OPT_KEY); ?>[checkout_suggestions]"><?php echo esc_textarea($s['checkout_suggestions']); ?></textarea><p class="description">One per line</p></td></tr>
                    <tr><th scope="row">Gemini API key</th><td><input type="password" class="regular-text" name="<?php echo esc_attr(self::OPT_KEY); ?>[gemini_api_key]" value="<?php echo esc_attr($s['gemini_api_key']); ?>" /></td></tr>
                    <tr><th scope="row">Gemini model</th><td><input type="text" class="regular-text" name="<?php echo esc_attr(self::OPT_KEY); ?>[gemini_model]" value="<?php echo esc_attr($s['gemini_model']); ?>" /><p class="description">e.g. gemini-1.5-flash</p></td></tr>
                    <tr><th scope="row">System prompt</th><td><textarea class="large-text" rows="4" name="<?php echo esc_attr(self::OPT_KEY); ?>[system_prompt]"><?php echo esc_textarea($s['system_prompt']); ?></textarea></td></tr>
                    <tr><th scope="row">Enable checkout assistance</th><td><label><input type="checkbox" name="<?php echo esc_attr(self::OPT_KEY); ?>[checkout_enable]" value="yes" <?php checked($s['checkout_enable'], 'yes'); ?>/> Enable shipping cutoff and policy answers on cart/checkout pages</label></td></tr>
                    <tr><th scope="row">Cutoff time (24h)</th><td><input type="text" class="regular-text" name="<?php echo esc_attr(self::OPT_KEY); ?>[cutoff_time]" value="<?php echo esc_attr($s['cutoff_time']); ?>" /><p class="description">Format HH:MM, e.g. 16:00</p></td></tr>
                    <tr><th scope="row">Cutoff days</th><td><input type="text" class="regular-text" name="<?php echo esc_attr(self::OPT_KEY); ?>[cutoff_days]" value="<?php echo esc_attr($s['cutoff_days']); ?>" /><p class="description">Comma separated numeric days 0=Sun … 6=Sat. Orders on other days will ship next working day.</p></td></tr>
                    <tr><th scope="row">Cutoff timezone</th><td><input type="text" class="regular-text" name="<?php echo esc_attr(self::OPT_KEY); ?>[cutoff_timezone]" value="<?php echo esc_attr($s['cutoff_timezone']); ?>" /><p class="description">Optional override. Leave blank to use site timezone.</p></td></tr>
                    <tr><th scope="row">Dispatch message (today)</th><td><input type="text" class="regular-text" name="<?php echo esc_attr(self::OPT_KEY); ?>[dispatch_today_msg]" value="<?php echo esc_attr($s['dispatch_today_msg']); ?>" /></td></tr>
                    <tr><th scope="row">Dispatch message (next working day)</th><td><input type="text" class="regular-text" name="<?php echo esc_attr(self::OPT_KEY); ?>[dispatch_next_msg]" value="<?php echo esc_attr($s['dispatch_next_msg']); ?>" /></td></tr>
                    <tr><th scope="row">Shipping policy text</th><td><textarea class="large-text" rows="3" name="<?php echo esc_attr(self::OPT_KEY); ?>[knowledge_shipping]"><?php echo esc_textarea($s['knowledge_shipping']); ?></textarea></td></tr>
                    <tr><th scope="row">Returns policy text</th><td><textarea class="large-text" rows="3" name="<?php echo esc_attr(self::OPT_KEY); ?>[knowledge_returns]"><?php echo esc_textarea($s['knowledge_returns']); ?></textarea></td></tr>
                    <tr><th scope="row">Contact information</th><td><textarea class="large-text" rows="2" name="<?php echo esc_attr(self::OPT_KEY); ?>[knowledge_contact]"><?php echo esc_textarea($s['knowledge_contact']); ?></textarea></td></tr>
                    <tr><th scope="row">License key</th><td><input type="text" class="regular-text" name="<?php echo esc_attr(self::OPT_KEY); ?>[license_key]" value="<?php echo esc_attr($s['license_key']); ?>" /><p class="description">Status: <?php echo empty($license['valid']) ? '<span style="color:#b00;">Inactive</span>' : '<span style="color:green;">Active</span>'; ?> <?php echo esc_html($license['message'] ?? ''); ?></p></td></tr>
                    <tr><th scope="row">License server URL</th><td><input type="text" class="regular-text" name="<?php echo esc_attr(self::OPT_KEY); ?>[license_server]" value="<?php echo esc_attr($s['license_server']); ?>" /></td></tr>
                    <tr><th scope="row">Product ID</th><td><input type="text" class="regular-text" name="<?php echo esc_attr(self::OPT_KEY); ?>[product_id]" value="<?php echo esc_attr($s['product_id']); ?>" /></td></tr>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }

    /**
     * License validation with caching
     */
    public function license_state($force = false) {
        $cached = get_transient(self::TRANS_LICENSE);
        if (!$force && is_array($cached)) {
            return $cached;
        }
        $s   = $this->get_settings();
        $key = trim((string) $s['license_key']);
        if ($key === '') {
            $state = ['valid' => false, 'message' => 'No license key set'];
            set_transient(self::TRANS_LICENSE, $state, HOUR_IN_SECONDS);
            return $state;
        }
        $payload = [
            'action'      => 'validate',
            'license_key' => $key,
            'domain'      => $this->current_domain(),
            'product_id'  => $s['product_id'],
        ];
        $resp = wp_remote_post($s['license_server'], [
            'timeout' => 15,
            'body'    => $payload,
        ]);
        if (is_wp_error($resp)) {
            $state = ['valid' => false, 'message' => 'License check failed'];
            set_transient(self::TRANS_LICENSE, $state, 15 * MINUTE_IN_SECONDS);
            return $state;
        }
        $code = wp_remote_retrieve_response_code($resp);
        $body = wp_remote_retrieve_body($resp);
        $json = json_decode($body, true);
        if ($code !== 200 || !is_array($json)) {
            $state = ['valid' => false, 'message' => 'License server error'];
            set_transient(self::TRANS_LICENSE, $state, 15 * MINUTE_IN_SECONDS);
            return $state;
        }
        $valid  = !empty($json['valid']) || !empty($json['success']);
        $state  = [
            'valid'   => (bool) $valid,
            'message' => (string) ($json['message'] ?? ($valid ? 'OK' : 'Invalid license')),
        ];
        set_transient(self::TRANS_LICENSE, $state, 6 * HOUR_IN_SECONDS);
        return $state;
    }

    private function current_domain() {
        $host = parse_url(home_url(), PHP_URL_HOST);
        return $host ? strtolower($host) : '';
    }

    /**
     * Ajax: search products
     */
    public function ajax_search_products() {
        check_ajax_referer('sacp_nonce', 'nonce');
        if (!class_exists('WooCommerce')) {
            wp_send_json_error(['message' => 'WooCommerce not active']);
        }
        $q   = isset($_POST['q']) ? sanitize_text_field((string) $_POST['q']) : '';
        $max = isset($_POST['max']) ? max(1, min(12, intval($_POST['max']))) : 6;
        $args = [
            'status'  => 'publish',
            'limit'   => $max,
            's'       => $q,
            'orderby' => 'relevance',
        ];
        $products = wc_get_products($args);
        $out = [];
        foreach ($products as $p) {
            $out[] = [
                'id'    => $p->get_id(),
                'name'  => $p->get_name(),
                'price_html' => $p->get_price_html(),
                'image' => wp_get_attachment_image_url($p->get_image_id(), 'woocommerce_thumbnail') ?: wc_placeholder_img_src(),
                'url'   => get_permalink($p->get_id()),
                'in_stock' => $p->is_in_stock(),
            ];
        }
        wp_send_json_success(['products' => $out]);
    }

    /**
     * Ajax: add to cart
     */
    public function ajax_add_to_cart() {
        check_ajax_referer('sacp_nonce', 'nonce');
        if (!class_exists('WooCommerce')) {
            wp_send_json_error(['message' => 'WooCommerce not active']);
        }
        $product_id = isset($_POST['product_id']) ? absint($_POST['product_id']) : 0;
        $qty        = isset($_POST['qty']) ? max(1, absint($_POST['qty'])) : 1;
        if (!$product_id) {
            wp_send_json_error(['message' => 'Missing product']);
        }
        $added = WC()->cart->add_to_cart($product_id, $qty);
        if (!$added) {
            wp_send_json_error(['message' => 'Could not add to cart']);
        }
        wp_send_json_success([
            'message'   => 'Added to cart',
            'cart_count'=> WC()->cart->get_cart_contents_count(),
            'cart_url'  => wc_get_cart_url(),
        ]);
    }

    /**
     * Compute checkout context for shipping cutoff and policy answers
     */
    private function get_checkout_context() {
        $s = $this->get_settings();
        $ctx = [];
        // shipping cutoff logic
        $tz_string = $s['cutoff_timezone'] ?: wp_timezone_string();
        try {
            $tz = new DateTimeZone($tz_string);
        } catch (Exception $e) {
            $tz = new DateTimeZone(wp_timezone_string());
        }
        $now = new DateTime('now', $tz);
        // parse cutoff time
        $cutoff = DateTime::createFromFormat('H:i', $s['cutoff_time'], $tz);
        if (!$cutoff) {
            $cutoff = DateTime::createFromFormat('H:i', '16:00', $tz);
        }
        // ensure cutoff date is today
        $cutoff->setDate((int)$now->format('Y'), (int)$now->format('m'), (int)$now->format('d'));
        // allowed days
        $days = array_map('trim', explode(',', $s['cutoff_days']));
        $current_day = (int) $now->format('w');
        $is_working_day = in_array((string) $current_day, $days, true);
        $ships_today = false;
        if ($is_working_day && $now <= $cutoff) {
            $ships_today = true;
        }
        $ctx['shipping_cutoff'] = [
            'now'           => $now->format(DateTimeInterface::ATOM),
            'cutoff_time'   => $cutoff->format('H:i'),
            'is_working_day'=> $is_working_day,
            'ships_today'   => $ships_today,
            'today_message' => $s['dispatch_today_msg'],
            'next_message'  => $s['dispatch_next_msg'],
        ];
        // policies
        $ctx['policy'] = [
            'shipping' => $s['knowledge_shipping'],
            'returns'  => $s['knowledge_returns'],
            'contact'  => $s['knowledge_contact'],
        ];
        return $ctx;
    }

    /**
     * Build shop context used in AI conversation
     */
    private function build_shop_context($user_text, $on_checkout) {
        $context = [
            'site'     => home_url(),
            'currency' => function_exists('get_woocommerce_currency') ? get_woocommerce_currency() : '',
            'language' => get_locale(),
        ];
        // Add top matches for browsing
        if (class_exists('WooCommerce') && strlen($user_text) >= 2 && !$on_checkout) {
            $products = wc_get_products([
                'status'  => 'publish',
                'limit'   => 4,
                's'       => $user_text,
                'orderby' => 'relevance',
            ]);
            $context['top_matches'] = array_map(function($p) {
                return [
                    'id'    => $p->get_id(),
                    'name'  => $p->get_name(),
                    'price_html' => $p->get_price_html(),
                    'url'   => get_permalink($p->get_id()),
                ];
            }, $products);
        }
        // include checkout assistance context if on checkout and enabled
        $s = $this->get_settings();
        if ($on_checkout && $s['checkout_enable'] === 'yes') {
            $context['checkout'] = $this->get_checkout_context();
        }
        return $context;
    }

    /**
     * Basic deterministic responses for checkout assistance when AI disabled
     */
    private function maybe_checkout_fallback_response($user_text, $on_checkout) {
        $s = $this->get_settings();
        if (!$on_checkout || $s['checkout_enable'] !== 'yes') {
            return null;
        }
        $text = strtolower($user_text);
        // shipping cutoff queries
        if (strpos($text, 'ship') !== false || strpos($text, 'shipping') !== false || strpos($text, 'cutoff') !== false) {
            $ctx = $this->get_checkout_context();
            $msg = $ctx['shipping_cutoff']['ships_today'] ? $ctx['shipping_cutoff']['today_message'] : $ctx['shipping_cutoff']['next_message'];
            return $msg;
        }
        // returns / refund queries
        if (strpos($text, 'return') !== false || strpos($text, 'refund') !== false) {
            return $s['knowledge_returns'];
        }
        // shipping policy
        if (strpos($text, 'shipping') !== false) {
            return $s['knowledge_shipping'];
        }
        // contact
        if (strpos($text, 'contact') !== false || strpos($text, 'support') !== false) {
            return $s['knowledge_contact'];
        }
        return null;
    }

    /**
     * Ajax: main chat endpoint
     */
    public function ajax_chat() {
        check_ajax_referer('sacp_nonce', 'nonce');
        $s = $this->get_settings();
        // detect on checkout
        $on_checkout = false;
        if (function_exists('is_cart') && function_exists('is_checkout')) {
            $on_checkout = is_cart() || is_checkout();
        }
        $messages = isset($_POST['messages']) && is_array($_POST['messages']) ? $_POST['messages'] : [];
        $user_text = isset($_POST['user_text']) ? sanitize_text_field((string) $_POST['user_text']) : '';
        // Fallback deterministic response for checkout assistance
        $fallback = $this->maybe_checkout_fallback_response($user_text, $on_checkout);
        // If fallback exists and AI is not active or not needed, send fallback
        $license = $this->license_state();
        $ai_enabled = !empty($license['valid']) && !empty($s['gemini_api_key']);
        if ($fallback && !$ai_enabled) {
            wp_send_json_success([
                'reply' => $fallback,
                'products_directive' => null,
            ]);
            return;
        }
        // If AI disabled and no deterministic fallback, send minimal reply
        if (!$ai_enabled) {
            wp_send_json_success([
                'reply' => $fallback ?: 'I can help with product suggestions or answer checkout questions.',
                'products_directive' => null,
            ]);
            return;
        }
        // Build context and call Gemini
        $context = $this->build_shop_context($user_text, $on_checkout);
        $reply = $this->call_gemini($s['gemini_api_key'], $s['gemini_model'], $s['system_prompt'], $context, $messages, $user_text, $on_checkout);
        if (is_wp_error($reply)) {
            wp_send_json_error(['message' => $reply->get_error_message()]);
        } else {
            wp_send_json_success($reply);
        }
    }

    /**
     * Call Gemini API via proxy
     */
    private function call_gemini($api_key, $model, $system_prompt, $shop_context, $messages, $user_text, $on_checkout) {
        $endpoint = 'https://generativelanguage.googleapis.com/v1beta/models/' . rawurlencode($model) . ':generateContent?key=' . rawurlencode($api_key);
        $history_text = '';
        foreach ($messages as $m) {
            if (!is_array($m)) continue;
            $role = ($m['role'] ?? '') === 'assistant' ? 'Assistant' : 'User';
            $content = sanitize_text_field((string) ($m['content'] ?? ''));
            if ($content !== '') {
                $history_text .= $role . ': ' . $content . "\n";
            }
        }
        $prompt = "SYSTEM:\n" . $system_prompt . "\n\n" .
            "SHOP CONTEXT (JSON):\n" . wp_json_encode($shop_context) . "\n\n" .
            "CHAT HISTORY:\n" . $history_text .
            "USER:\n" . $user_text . "\n\n";
        $instructions = "INSTRUCTIONS:\n";
        if ($on_checkout) {
            $instructions .= "If the user asks about shipping or cutoff times, answer using the provided checkout.shipping_cutoff data. If they ask about returns or contact info, answer using checkout.policy.\n";
        }
        $instructions .= "If recommending products, return a short answer then output a JSON block on a new line starting with SACP_PRODUCTS: containing an array of product search keywords. Example: SACP_PRODUCTS: {\"q\":\"running shoes\", \"max\":6}. Keep responses concise.";
        $prompt .= $instructions;
        $body = [
            'contents' => [
                ['parts' => [ ['text' => $prompt] ]],
            ],
            'generationConfig' => [
                'temperature'     => 0.6,
                'maxOutputTokens' => 400,
            ],
        ];
        $resp = wp_remote_post($endpoint, [
            'timeout' => 25,
            'headers' => [ 'Content-Type' => 'application/json' ],
            'body'    => wp_json_encode($body),
        ]);
        if (is_wp_error($resp)) {
            return $resp;
        }
        $code = wp_remote_retrieve_response_code($resp);
        $raw  = wp_remote_retrieve_body($resp);
        $json = json_decode($raw, true);
        if ($code !== 200 || !is_array($json)) {
            return new WP_Error('gemini_error', 'Gemini request failed');
        }
        $text = $json['candidates'][0]['content']['parts'][0]['text'] ?? '';
        $text = is_string($text) ? $text : '';
        // detect optional product search directive
        $products_directive = null;
        if (preg_match('/SACP_PRODUCTS:\s*(\{.*\})/s', $text, $m)) {
            $products_directive = json_decode($m[1], true);
            $text = trim(str_replace($m[0], '', $text));
        }
        return [
            'reply' => $text ?: 'OK.',
            'products_directive' => $products_directive,
        ];
    }
}

// initialize
Sparkcut_AI_Chatbot_Pro::instance();