<?php
/**
 * Plugin Name: WooCommerce Low Stock Nudge
 * Plugin URI: https://your-site.com
 * Description: Show urgency nudges like “Only X left in stock” on WooCommerce products. Pro unlocks per-product overrides and advanced urgency logic.
 * Version: 1.0.0
 * Author: Sparkcut Labs
 * Author URI: https://sparkcutlabs.com
 * Text Domain: wclsnp
 * Requires PHP: 7.4
 * Requires at least: 5.8
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'WCLSNP_VERSION', '1.0.0' );
define( 'WCLSNP_FILE', __FILE__ );
define( 'WCLSNP_DIR', plugin_dir_path( __FILE__ ) );
define( 'WCLSNP_URL', plugin_dir_url( __FILE__ ) );

// License server endpoint (replace with your real URL, e.g. https://example.com/license/api.php).
define( 'WCLSNP_LICENSE_API', 'http://sparkcutlabs.com/license-server/api.php' );

// Product slug as used in your license server.
define( 'WCLSNP_PRODUCT_SLUG', 'low_stock_nudge_pro' );

/**
 * Unified License Client for Low Stock Nudge
 *
 * - Talks to Sparkcut Labs (your) license server
 * - Caches validation result in transients
 * - Provides WCLSNP_License::is_pro() for feature gating
 */
class WCLSNP_License {

    const OPT_KEY         = 'wclsnp_license';
    const TRANSIENT_STATE = 'wclsnp_lic_status'; // 'valid', 'invalid', 'grace', etc.
    const GRACE_HOURS     = 72;
    const CHECK_INTERVAL  = 12 * HOUR_IN_SECONDS;

    /**
     * Bootstrap admin hooks.
     */
    public function init() {
        if ( is_admin() ) {
            add_action( 'admin_menu', [ $this, 'register_menu' ] );
            add_action( 'admin_init', [ $this, 'maybe_handle_form' ] );
        }
    }

    /**
     * True if license is currently valid (or in grace period).
     *
     * @return bool
     */
    public static function is_pro() {
        $status = self::cached_status();
        if ( $status === 'valid' ) {
            return true;
        }
        if ( $status === 'grace' ) {
            return true;
        }
        return false;
    }

    /**
     * Get cached status from transient (or revalidate if stale).
     *
     * @return string 'valid'|'invalid'|'grace'| 'unknown'
     */
    public static function cached_status() {
        $status = get_transient( self::TRANSIENT_STATE );
        if ( $status ) {
            return $status;
        }

        // No cached status: try background validation (no error output).
        $opt = get_option(
            self::OPT_KEY,
            [
                'key'         => '',
                'activated'   => false,
                'grace_until' => 0,
                'last_check'  => 0,
            ]
        );

        if ( empty( $opt['key'] ) ) {
            set_transient( self::TRANSIENT_STATE, 'invalid', HOUR_IN_SECONDS );
            return 'invalid';
        }

        // If last_check is recent, infer status from grace_until.
        $now = time();
        if ( ! empty( $opt['last_check'] ) && ( $now - (int) $opt['last_check'] ) < self::CHECK_INTERVAL ) {
            if ( ! empty( $opt['activated'] ) ) {
                set_transient( self::TRANSIENT_STATE, 'valid', HOUR_IN_SECONDS );
                return 'valid';
            }
            if ( ! empty( $opt['grace_until'] ) && $opt['grace_until'] > $now ) {
                set_transient( self::TRANSIENT_STATE, 'grace', HOUR_IN_SECONDS );
                return 'grace';
            }
            set_transient( self::TRANSIENT_STATE, 'invalid', HOUR_IN_SECONDS );
            return 'invalid';
        }

        // Silent background validate.
        $self = new self();
        $res  = $self->remote_request( $opt['key'], 'validate' );
        $self->handle_validation_result( $res, $opt, false );

        $status = get_transient( self::TRANSIENT_STATE );
        return $status ? $status : 'unknown';
    }

    /**
     * Register menu item under WooCommerce.
     */
    public function register_menu() {
        add_submenu_page(
            'woocommerce',
            __( 'Low Stock Nudge', 'wclsnp' ),
            __( 'Low Stock Nudge', 'wclsnp' ),
            'manage_woocommerce',
            'wclsnp-settings',
            [ $this, 'render_settings_page' ]
        );
    }

    /**
     * Process license + settings form.
     */
    public function maybe_handle_form() {
        if ( empty( $_POST['wclsnp_settings_nonce'] ) ) {
            return;
        }

        if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['wclsnp_settings_nonce'] ) ), 'wclsnp_save_settings' ) ) {
            return;
        }

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        // Save general settings.
        $settings = [
            'threshold'             => isset( $_POST['wclsnp_threshold'] ) ? max( 1, (int) $_POST['wclsnp_threshold'] ) : 5,
            'message'               => isset( $_POST['wclsnp_message'] ) ? wp_kses_post( wp_unslash( $_POST['wclsnp_message'] ) ) : __( 'Hurry! Only {stock} left in stock.', 'wclsnp' ),
            'show_on_product'       => ! empty( $_POST['wclsnp_show_on_product'] ) ? 1 : 0,
            'show_on_archive'       => ! empty( $_POST['wclsnp_show_on_archive'] ) ? 1 : 0,
            'show_on_cart'          => ! empty( $_POST['wclsnp_show_on_cart'] ) ? 1 : 0,
            'show_on_checkout'      => ! empty( $_POST['wclsnp_show_on_checkout'] ) ? 1 : 0,
            'enable_dynamic_urgency'=> ! empty( $_POST['wclsnp_enable_dynamic_urgency'] ) ? 1 : 0,
        ];
        update_option( 'wclsnp_settings', $settings );

        // Handle license key (if submitted).
        if ( isset( $_POST['wclsnp_license_key'] ) ) {
            $key = trim( sanitize_text_field( wp_unslash( $_POST['wclsnp_license_key'] ) ) );
            $opt = get_option(
                self::OPT_KEY,
                [
                    'key'         => '',
                    'activated'   => false,
                    'grace_until' => 0,
                    'last_check'  => 0,
                ]
            );

            if ( $key !== $opt['key'] ) {
                // New key: attempt activation.
                $res = $this->remote_request( $key, 'activate' );
                $this->handle_validation_result( $res, $opt, true, $key );
            } else {
                // Same key: re-validate.
                $res = $this->remote_request( $key, 'validate' );
                $this->handle_validation_result( $res, $opt, true );
            }
        }

        wp_safe_redirect(
            add_query_arg(
                [
                    'page'      => 'wclsnp-settings',
                    'updated'   => 'true',
                ],
                admin_url( 'admin.php' )
            )
        );
        exit;
    }

    /**
     * Render combined settings + license page.
     */
    public function render_settings_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        $settings = get_option(
            'wclsnp_settings',
            [
                'threshold'             => 5,
                'message'               => __( 'Hurry! Only {stock} left in stock.', 'wclsnp' ),
                'show_on_product'       => 1,
                'show_on_archive'       => 0,
                'show_on_cart'          => 0,
                'show_on_checkout'      => 0,
                'enable_dynamic_urgency'=> 0,
            ]
        );

        $opt    = get_option(
            self::OPT_KEY,
            [
                'key'         => '',
                'activated'   => false,
                'grace_until' => 0,
                'last_check'  => 0,
            ]
        );
        $status = self::cached_status();
        $domain = $this->domain();
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Low Stock Nudge Settings', 'wclsnp' ); ?></h1>

            <?php if ( isset( $_GET['updated'] ) ) : ?>
                <div class="notice notice-success"><p><?php esc_html_e( 'Settings saved.', 'wclsnp' ); ?></p></div>
            <?php endif; ?>

            <form method="post">
                <?php wp_nonce_field( 'wclsnp_save_settings', 'wclsnp_settings_nonce' ); ?>

                <h2><?php esc_html_e( 'General', 'wclsnp' ); ?></h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="wclsnp_threshold"><?php esc_html_e( 'Global low stock threshold', 'wclsnp' ); ?></label></th>
                        <td>
                            <input type="number" min="1" id="wclsnp_threshold" name="wclsnp_threshold" value="<?php echo esc_attr( $settings['threshold'] ); ?>" />
                            <p class="description"><?php esc_html_e( 'Show the nudge when stock quantity is less than or equal to this number.', 'wclsnp' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wclsnp_message"><?php esc_html_e( 'Default message', 'wclsnp' ); ?></label></th>
                        <td>
                            <textarea id="wclsnp_message" name="wclsnp_message" rows="3" cols="60"><?php echo esc_textarea( $settings['message'] ); ?></textarea>
                            <p class="description"><?php esc_html_e( 'Use {stock} for remaining quantity, {product} for product name.', 'wclsnp' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Display locations', 'wclsnp' ); ?></th>
                        <td>
                            <label><input type="checkbox" name="wclsnp_show_on_product" value="1" <?php checked( $settings['show_on_product'], 1 ); ?> /> <?php esc_html_e( 'Single product page', 'wclsnp' ); ?></label><br/>
                            <label><input type="checkbox" name="wclsnp_show_on_archive" value="1" <?php checked( $settings['show_on_archive'], 1 ); ?> /> <?php esc_html_e( 'Shop & category archives', 'wclsnp' ); ?></label><br/>
                            <label><input type="checkbox" name="wclsnp_show_on_cart" value="1" <?php checked( $settings['show_on_cart'], 1 ); ?> /> <?php esc_html_e( 'Cart page (Pro)', 'wclsnp' ); ?></label><br/>
                            <label><input type="checkbox" name="wclsnp_show_on_checkout" value="1" <?php checked( $settings['show_on_checkout'], 1 ); ?> /> <?php esc_html_e( 'Checkout page (Pro)', 'wclsnp' ); ?></label>
                            <?php if ( ! self::is_pro() ) : ?>
                                <p class="description"><em><?php esc_html_e( 'Cart and Checkout positions require an active Pro license.', 'wclsnp' ); ?></em></p>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Dynamic urgency (Pro)', 'wclsnp' ); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wclsnp_enable_dynamic_urgency" value="1" <?php checked( $settings['enable_dynamic_urgency'], 1 ); ?> />
                                <?php esc_html_e( 'Estimate days of stock remaining based on recent sales.', 'wclsnp' ); ?>
                            </label>
                            <?php if ( ! self::is_pro() ) : ?>
                                <p class="description"><em><?php esc_html_e( 'Requires an active Pro license.', 'wclsnp' ); ?></em></p>
                            <?php else : ?>
                                <p class="description"><?php esc_html_e( 'On supported products, the message will include an approximate number of days before this item is sold out.', 'wclsnp' ); ?></p>
                            <?php endif; ?>
                        </td>
                    </tr>
                </table>

                <hr/>

                <h2><?php esc_html_e( 'License', 'wclsnp' ); ?></h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Current status', 'wclsnp' ); ?></th>
                        <td>
                            <?php
                            if ( $status === 'valid' ) {
                                echo '<span style="color:green;font-weight:bold;">' . esc_html__( 'Active', 'wclsnp' ) . '</span>';
                            } elseif ( $status === 'grace' ) {
                                echo '<span style="color:#e6a700;font-weight:bold;">' . esc_html__( 'Grace period', 'wclsnp' ) . '</span>';
                            } else {
                                echo '<span style="color:#cc0000;font-weight:bold;">' . esc_html__( 'Inactive', 'wclsnp' ) . '</span>';
                            }
                            ?>
                            <p class="description">
                                <?php
                                printf(
                                    esc_html__( 'License is tied to this domain: %s', 'wclsnp' ),
                                    '<code>' . esc_html( $domain ) . '</code>'
                                );
                                ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wclsnp_license_key"><?php esc_html_e( 'License key', 'wclsnp' ); ?></label></th>
                        <td>
                            <input type="text" id="wclsnp_license_key" name="wclsnp_license_key" value="<?php echo esc_attr( $opt['key'] ); ?>" class="regular-text" />
                            <p class="description"><?php esc_html_e( 'Enter the license key you received after purchase.', 'wclsnp' ); ?></p>
                        </td>
                    </tr>
                </table>

                <?php submit_button( __( 'Save settings', 'wclsnp' ) ); ?>
            </form>
        </div>
        <?php
    }

    /**
     * Perform remote request to license server.
     *
     * @param string $key
     * @param string $action 'validate'|'activate'
     * @return array [ 'ok'=>bool, 'msg'=>string, 'active'=>bool ]
     */
    private function remote_request( $key, $action ) {
        $body = [
            'action'      => $action,
            'license_key' => $key,
            'product'     => WCLSNP_PRODUCT_SLUG,
            'domain'      => $this->domain(),
        ];

        $response = wp_remote_post(
            WCLSNP_LICENSE_API,
            [
                'timeout' => 10,
                'body'    => $body,
            ]
        );

        if ( is_wp_error( $response ) ) {
            return [
                'ok'     => false,
                'msg'    => $response->get_error_message(),
                'active' => false,
            ];
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        if ( $code !== 200 ) {
            return [
                'ok'     => false,
                'msg'    => 'HTTP ' . $code,
                'active' => false,
            ];
        }

        $data = json_decode( $body, true );
        if ( ! is_array( $data ) ) {
            return [
                'ok'     => false,
                'msg'    => 'Invalid JSON from server',
                'active' => false,
            ];
        }

        $data = wp_parse_args(
            $data,
            [
                'ok'     => false,
                'msg'    => '',
                'active' => true,
            ]
        );

        return $data;
    }

    /**
     * Update options + transients based on server response.
     *
     * @param array $res
     * @param array $opt
     * @param bool  $show_admin_notice
     * @param null|string $new_key
     */
    private function handle_validation_result( $res, $opt, $show_admin_notice = false, $new_key = null ) {
        $now = time();

        if ( $new_key !== null ) {
            $opt['key'] = $new_key;
        }

        if ( ! empty( $res['ok'] ) && ! empty( $res['active'] ) ) {
            $opt['activated']   = true;
            $opt['grace_until'] = 0;
            $opt['last_check']  = $now;
            update_option( self::OPT_KEY, $opt );
            set_transient( self::TRANSIENT_STATE, 'valid', self::CHECK_INTERVAL );

            if ( $show_admin_notice ) {
                add_action(
                    'admin_notices',
                    function() use ( $res ) {
                        echo '<div class="notice notice-success"><p>' . esc_html( $res['msg'] ?? __( 'License validated.', 'wclsnp' ) ) . '</p></div>';
                    }
                );
            }

            return;
        }

        // Not ok or not active: start/continue grace period.
        $opt['activated']  = false;
        $opt['last_check'] = $now;

        if ( empty( $opt['grace_until'] ) || $opt['grace_until'] < $now ) {
            $opt['grace_until'] = $now + ( self::GRACE_HOURS * HOUR_IN_SECONDS );
        }

        update_option( self::OPT_KEY, $opt );

        if ( $opt['grace_until'] > $now ) {
            set_transient( self::TRANSIENT_STATE, 'grace', self::CHECK_INTERVAL );
        } else {
            set_transient( self::TRANSIENT_STATE, 'invalid', self::CHECK_INTERVAL );
        }

        if ( $show_admin_notice ) {
            $msg = ! empty( $res['msg'] ) ? $res['msg'] : __( 'License validation failed.', 'wclsnp' );
            add_action(
                'admin_notices',
                function() use ( $msg ) {
                    echo '<div class="notice notice-warning"><p>' . esc_html( $msg ) . '</p></div>';
                }
            );
        }
    }

    /**
     * Domain we bind the license to.
     *
     * @return string
     */
    private function domain() {
        $home = home_url();
        $host = wp_parse_url( $home, PHP_URL_HOST );
        return $host ? $host : $home;
    }
}

/**
 * Main Low Stock Nudge functionality (frontend).
 */
class WCLSNP_Plugin {

    public function init() {
        if ( ! class_exists( 'WooCommerce' ) ) {
            return;
        }

        add_action( 'add_meta_boxes', [ $this, 'register_product_metabox' ] );
        add_action( 'save_post_product', [ $this, 'save_product_meta' ], 10, 2 );

        $settings = get_option(
            'wclsnp_settings',
            [
                'threshold'             => 5,
                'message'               => __( 'Hurry! Only {stock} left in stock.', 'wclsnp' ),
                'show_on_product'       => 1,
                'show_on_archive'       => 0,
                'show_on_cart'          => 0,
                'show_on_checkout'      => 0,
                'enable_dynamic_urgency'=> 0,
            ]
        );

        if ( ! empty( $settings['show_on_product'] ) ) {
            add_action( 'woocommerce_single_product_summary', [ $this, 'output_nudge_on_single' ], 25 );
        }
        if ( ! empty( $settings['show_on_archive'] ) ) {
            add_action( 'woocommerce_after_shop_loop_item_title', [ $this, 'output_nudge_on_archive' ], 15 );
        }

        // Cart / checkout nudges are Pro-only.
        if ( WCLSNP_License::is_pro() ) {
            if ( ! empty( $settings['show_on_cart'] ) ) {
                add_action( 'woocommerce_cart_totals_before_order_total', [ $this, 'output_nudges_on_cart' ] );
            }
            if ( ! empty( $settings['show_on_checkout'] ) ) {
                add_action( 'woocommerce_review_order_before_payment', [ $this, 'output_nudges_on_checkout' ] );
            }
        }
    }

    /**
     * Add Pro metabox for per-product overrides.
     */
    public function register_product_metabox() {
        add_meta_box(
            'wclsnp_metabox',
            __( 'Low Stock Nudge (Pro)', 'wclsnp' ),
            [ $this, 'render_product_metabox' ],
            'product',
            'side',
            'default'
        );
    }

    public function render_product_metabox( $post ) {
        wp_nonce_field( 'wclsnp_save_meta', 'wclsnp_meta_nonce' );

        $threshold = get_post_meta( $post->ID, '_wclsnp_threshold', true );
        $message   = get_post_meta( $post->ID, '_wclsnp_message', true );

        if ( ! WCLSNP_License::is_pro() ) {
            echo '<p><em>' . esc_html__( 'Per-product overrides are available with a Pro license.', 'wclsnp' ) . '</em></p>';
            echo '<p>' . esc_html__( 'This product will use the global threshold and message from the Low Stock Nudge settings page.', 'wclsnp' ) . '</p>';
            return;
        }
        ?>
        <p>
            <label for="wclsnp_threshold"><?php esc_html_e( 'Custom low stock threshold', 'wclsnp' ); ?></label><br/>
            <input type="number" min="1" id="wclsnp_threshold" name="wclsnp_threshold" value="<?php echo esc_attr( $threshold ); ?>" style="width:100%;" />
            <span class="description"><?php esc_html_e( 'Optional. Leave empty to use global threshold.', 'wclsnp' ); ?></span>
        </p>
        <p>
            <label for="wclsnp_message"><?php esc_html_e( 'Custom message', 'wclsnp' ); ?></label><br/>
            <textarea id="wclsnp_message" name="wclsnp_message" rows="3" style="width:100%;"><?php echo esc_textarea( $message ); ?></textarea>
            <span class="description"><?php esc_html_e( 'Optional. Use {stock} and {product}. Leave empty to use global message.', 'wclsnp' ); ?></span>
        </p>
        <?php
    }

    public function save_product_meta( $post_id, $post ) {
        if ( ! isset( $_POST['wclsnp_meta_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['wclsnp_meta_nonce'] ) ), 'wclsnp_save_meta' ) ) {
            return;
        }

        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        if ( $post->post_type !== 'product' ) {
            return;
        }

        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        if ( WCLSNP_License::is_pro() ) {
            $threshold = isset( $_POST['wclsnp_threshold'] ) ? (int) $_POST['wclsnp_threshold'] : '';
            $message   = isset( $_POST['wclsnp_message'] ) ? wp_kses_post( wp_unslash( $_POST['wclsnp_message'] ) ) : '';

            if ( $threshold > 0 ) {
                update_post_meta( $post_id, '_wclsnp_threshold', $threshold );
            } else {
                delete_post_meta( $post_id, '_wclsnp_threshold' );
            }

            if ( $message !== '' ) {
                update_post_meta( $post_id, '_wclsnp_message', $message );
            } else {
                delete_post_meta( $post_id, '_wclsnp_message' );
            }
        }
    }

    /**
     * Nudge on single product page.
     */
    public function output_nudge_on_single() {
        global $product;
        if ( ! $product instanceof WC_Product ) {
            return;
        }
        echo $this->build_nudge_html( $product );
    }

    /**
     * Nudge on archive (loop) items.
     */
    public function output_nudge_on_archive() {
        global $product;
        if ( ! $product instanceof WC_Product ) {
            return;
        }
        echo $this->build_nudge_html( $product, 'archive' );
    }

    /**
     * Nudge on cart page (Pro).
     */
    public function output_nudges_on_cart() {
        if ( ! WC()->cart ) {
            return;
        }
        echo '<div class="wclsnp-cart-nudges">';
        foreach ( WC()->cart->get_cart() as $item ) {
            if ( empty( $item['data'] ) || ! $item['data'] instanceof WC_Product ) {
                continue;
            }
            echo $this->build_nudge_html( $item['data'], 'cart' );
        }
        echo '</div>';
    }

    /**
     * Nudge on checkout (Pro).
     */
    public function output_nudges_on_checkout() {
        if ( ! WC()->cart ) {
            return;
        }
        echo '<div class="wclsnp-checkout-nudges">';
        foreach ( WC()->cart->get_cart() as $item ) {
            if ( empty( $item['data'] ) || ! $item['data'] instanceof WC_Product ) {
                continue;
            }
            echo $this->build_nudge_html( $item['data'], 'checkout' );
        }
        echo '</div>';
    }

    /**
     * Main renderer for the nudge.
     *
     * @param WC_Product $product
     * @param string     $context single|archive|cart|checkout
     * @return string
     */
    private function build_nudge_html( WC_Product $product, $context = 'single' ) {
        if ( ! $product->managing_stock() ) {
            return '';
        }

        $stock = $product->get_stock_quantity();
        if ( $stock === null || $stock <= 0 ) {
            return '';
        }

        $settings = get_option(
            'wclsnp_settings',
            [
                'threshold'             => 5,
                'message'               => __( 'Hurry! Only {stock} left in stock.', 'wclsnp' ),
                'enable_dynamic_urgency'=> 0,
            ]
        );

        $threshold = (int) $settings['threshold'];

        // Per-product override (Pro only).
        if ( WCLSNP_License::is_pro() ) {
            $custom_threshold = (int) get_post_meta( $product->get_id(), '_wclsnp_threshold', true );
            if ( $custom_threshold > 0 ) {
                $threshold = $custom_threshold;
            }
        }

        if ( $stock > $threshold ) {
            return '';
        }

        $message = $settings['message'];
        if ( WCLSNP_License::is_pro() ) {
            $custom_msg = get_post_meta( $product->get_id(), '_wclsnp_message', true );
            if ( $custom_msg !== '' ) {
                $message = $custom_msg;
            }
        }

        $replacements = [
            '{stock}'   => (string) $stock,
            '{product}' => $product->get_name(),
        ];

        // Dynamic urgency (Pro).
        if ( WCLSNP_License::is_pro() && ! empty( $settings['enable_dynamic_urgency'] ) ) {
            $days_left = $this->estimate_days_left( $product );
            if ( $days_left !== null ) {
                $replacements['{days_left}'] = (string) $days_left;
                if ( strpos( $message, '{days_left}' ) === false ) {
                    // Append if merchant didn't add placeholder.
                    $message .= ' ' . sprintf(
                        /* translators: %d: days left */
                        __( '(Around %d days of stock remaining.)', 'wclsnp' ),
                        $days_left
                    );
                }
            }
        }

        $message = strtr( $message, $replacements );

        $classes = [
            'wclsnp-nudge',
            'wclsnp-context-' . sanitize_html_class( $context ),
        ];

        ob_start();
        ?>
        <div class="<?php echo esc_attr( implode( ' ', $classes ) ); ?>" style="background:#fff7e6;border:1px solid #f0c36d;border-radius:4px;padding:6px 10px;margin:6px 0;font-size:0.9em;">
            <span style="color:#8a6d3b;"><?php echo wp_kses_post( $message ); ?></span>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Estimate days of stock remaining based on last 7 days of sales.
     *
     * @param WC_Product $product
     * @return int|null
     */
    private function estimate_days_left( WC_Product $product ) {
        $product_id = $product->get_id();

        // Basic, light-weight estimation.
        $args = [
            'limit'        => 20,
            'type'         => 'shop_order',
            'status'       => [ 'wc-completed', 'wc-processing' ],
            'date_created' => '>' . ( time() - 7 * DAY_IN_SECONDS ),
            'return'       => 'ids',
        ];

        $orders = wc_get_orders( $args );
        if ( empty( $orders ) ) {
            return null;
        }

        $qty_sold = 0;
        foreach ( $orders as $order_id ) {
            $order = wc_get_order( $order_id );
            if ( ! $order ) {
                continue;
            }
            foreach ( $order->get_items() as $item ) {
                if ( $item->get_product_id() == $product_id ) {
                    $qty_sold += (float) $item->get_quantity();
                }
            }
        }

        if ( $qty_sold <= 0 ) {
            return null;
        }

        $stock = (float) $product->get_stock_quantity();
        if ( $stock <= 0 ) {
            return null;
        }

        $avg_per_day = $qty_sold / 7.0;
        if ( $avg_per_day <= 0 ) {
            return null;
        }

        $days_left = (int) ceil( $stock / $avg_per_day );
        if ( $days_left < 1 ) {
            $days_left = 1;
        }

        return $days_left;
    }
}

// Bootstrap: License + plugin logic.
add_action(
    'plugins_loaded',
    function() {
        // License client.
        $lic = new WCLSNP_License();
        $lic->init();

        // Frontend logic.
        $core = new WCLSNP_Plugin();
        $core->init();
    }
);
