<?php
/**
 * Plugin Name: WooCommerce One-Click Coupon
 * Plugin URI: https://your-site.com
 * Description: Add a one-click button that applies a coupon on cart/checkout. Pro unlocks best-coupon selection, savings preview, and advanced targeting.
 * Version: 1.0.0
 * Author: Sparkcut Labs
 * Author URI: https://sparkcutlabs.com
 * Text Domain: wcoct
 * Requires PHP: 7.4
 * Requires at least: 5.8
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'WCOCT_VER', '1.0.0' );
define( 'WCOCT_FILE', __FILE__ );
define( 'WCOCT_DIR', plugin_dir_path( __FILE__ ) );
define( 'WCOCT_URL', plugin_dir_url( __FILE__ ) );

// License server endpoint (Sparkcut Labs license server api.php).
// Example: https://your-licenses-domain.com/api.php
define( 'WCOCT_LICENSE_API', 'http://sparkcutlabs.com/license-server/api.php' );

// Product slug as used in your license server (licenses.json -> product).
define( 'WCOCT_PRODUCT_SLUG', 'oneclick_coupon_pro' );

/**
 * License manager for One-Click Coupon (Sparkcut Labs license server compatible).
 *
 * - Single-site license bound to domain
 * - Uses actions: activate, validate
 * - 72-hour grace window on validation failure
 */
class WCOCT_License {

    const OPT_KEY         = 'wcoct_license';
    const TRANSIENT_STATE = 'wcoct_license_status';
    const CHECK_INTERVAL  = 12 * HOUR_IN_SECONDS;
    const GRACE_HOURS     = 72;

    /**
     * Bootstrap admin UI & handlers.
     */
    public function init() {
        if ( is_admin() ) {
            add_action( 'admin_menu', [ $this, 'register_menu' ] );
            add_action( 'admin_init', [ $this, 'maybe_handle_form' ] );
        }
    }

    /**
     * Is Pro currently active?
     *
     * @return bool
     */
    public static function is_pro() {
        $status = self::get_cached_status();
        return in_array( $status, [ 'valid', 'grace' ], true );
    }

    /**
     * Get cached status, refreshing silently if needed.
     *
     * @return string valid|invalid|grace|unknown
     */
    public static function get_cached_status() {
        $status = get_transient( self::TRANSIENT_STATE );
        if ( $status ) {
            return $status;
        }

        $opt = get_option(
            self::OPT_KEY,
            [
                'key'         => '',
                'activated'   => false,
                'grace_until' => 0,
                'last_check'  => 0,
            ]
        );

        if ( empty( $opt['key'] ) ) {
            set_transient( self::TRANSIENT_STATE, 'invalid', HOUR_IN_SECONDS );
            return 'invalid';
        }

        $now = time();
        if ( ! empty( $opt['last_check'] ) && ( $now - (int) $opt['last_check'] ) < self::CHECK_INTERVAL ) {
            if ( ! empty( $opt['activated'] ) ) {
                set_transient( self::TRANSIENT_STATE, 'valid', HOUR_IN_SECONDS );
                return 'valid';
            }
            if ( ! empty( $opt['grace_until'] ) && $opt['grace_until'] > $now ) {
                set_transient( self::TRANSIENT_STATE, 'grace', HOUR_IN_SECONDS );
                return 'grace';
            }
            set_transient( self::TRANSIENT_STATE, 'invalid', HOUR_IN_SECONDS );
            return 'invalid';
        }

        // Silent background validation.
        $self = new self();
        $res  = $self->remote_request( $opt['key'], 'validate' );
        $self->handle_validation_result( $res, $opt, false );

        $status = get_transient( self::TRANSIENT_STATE );
        return $status ? $status : 'unknown';
    }

    /**
     * Add settings / license page under WooCommerce.
     */
    public function register_menu() {
        add_submenu_page(
            'woocommerce',
            __( 'One-Click Coupon', 'wcoct' ),
            __( 'One-Click Coupon', 'wcoct' ),
            'manage_woocommerce',
            'wcoct-settings',
            [ $this, 'render_settings_page' ]
        );
    }

    /**
     * Handle settings + license key submission.
     */
    public function maybe_handle_form() {
        if ( empty( $_POST['wcoct_settings_nonce'] ) ) {
            return;
        }

        if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['wcoct_settings_nonce'] ) ), 'wcoct_save_settings' ) ) {
            return;
        }

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        // Save main settings.
        $settings = [
            'global_coupon'        => isset( $_POST['wcoct_global_coupon'] ) ? strtoupper( sanitize_text_field( wp_unslash( $_POST['wcoct_global_coupon'] ) ) ) : '',
            'button_text_cart'     => isset( $_POST['wcoct_button_text_cart'] ) ? sanitize_text_field( wp_unslash( $_POST['wcoct_button_text_cart'] ) ) : __( 'Apply discount', 'wcoct' ),
            'button_text_checkout' => isset( $_POST['wcoct_button_text_checkout'] ) ? sanitize_text_field( wp_unslash( $_POST['wcoct_button_text_checkout'] ) ) : __( 'Apply discount', 'wcoct' ),
            'show_on_cart'         => ! empty( $_POST['wcoct_show_on_cart'] ) ? 1 : 0,
            'show_on_checkout'     => ! empty( $_POST['wcoct_show_on_checkout'] ) ? 1 : 0,
            'show_on_mini_cart'    => ! empty( $_POST['wcoct_show_on_mini_cart'] ) ? 1 : 0,
            'min_cart_total'       => isset( $_POST['wcoct_min_cart_total'] ) ? floatval( $_POST['wcoct_min_cart_total'] ) : 0.0,
            'roles_limit'          => isset( $_POST['wcoct_roles_limit'] ) && is_array( $_POST['wcoct_roles_limit'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['wcoct_roles_limit'] ) ) : [],
            // Pro-specific:
            'enable_auto_best'     => ! empty( $_POST['wcoct_enable_auto_best'] ) ? 1 : 0,
            'candidate_coupons'    => isset( $_POST['wcoct_candidate_coupons'] ) ? sanitize_textarea_field( wp_unslash( $_POST['wcoct_candidate_coupons'] ) ) : '',
            'show_savings_preview' => ! empty( $_POST['wcoct_show_savings_preview'] ) ? 1 : 0,
        ];
        update_option( 'wcoct_settings', $settings );

        // License key handling.
        if ( isset( $_POST['wcoct_license_key'] ) ) {
            $key = trim( sanitize_text_field( wp_unslash( $_POST['wcoct_license_key'] ) ) );
            $opt = get_option(
                self::OPT_KEY,
                [
                    'key'         => '',
                    'activated'   => false,
                    'grace_until' => 0,
                    'last_check'  => 0,
                ]
            );

            if ( $key !== $opt['key'] ) {
                // New key: activate.
                $res = $this->remote_request( $key, 'activate' );
                $this->handle_validation_result( $res, $opt, true, $key );
            } else {
                // Same key: validate.
                $res = $this->remote_request( $key, 'validate' );
                $this->handle_validation_result( $res, $opt, true );
            }
        }

        wp_safe_redirect(
            add_query_arg(
                [
                    'page'    => 'wcoct-settings',
                    'updated' => 'true',
                ],
                admin_url( 'admin.php' )
            )
        );
        exit;
    }

    /**
     * Render settings + license page.
     */
    public function render_settings_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        $settings = get_option(
            'wcoct_settings',
            [
                'global_coupon'        => '',
                'button_text_cart'     => __( 'Apply discount', 'wcoct' ),
                'button_text_checkout' => __( 'Apply discount', 'wcoct' ),
                'show_on_cart'         => 1,
                'show_on_checkout'     => 1,
                'show_on_mini_cart'    => 0,
                'min_cart_total'       => 0.0,
                'roles_limit'          => [],
                'enable_auto_best'     => 0,
                'candidate_coupons'    => '',
                'show_savings_preview' => 0,
            ]
        );

        $opt    = get_option(
            self::OPT_KEY,
            [
                'key'         => '',
                'activated'   => false,
                'grace_until' => 0,
                'last_check'  => 0,
            ]
        );
        $status = self::get_cached_status();
        $domain = $this->domain();

        $all_roles = wp_roles()->roles;
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'One-Click Coupon Settings', 'wcoct' ); ?></h1>

            <?php if ( isset( $_GET['updated'] ) ) : ?>
                <div class="notice notice-success"><p><?php esc_html_e( 'Settings saved.', 'wcoct' ); ?></p></div>
            <?php endif; ?>

            <form method="post">
                <?php wp_nonce_field( 'wcoct_save_settings', 'wcoct_settings_nonce' ); ?>

                <h2><?php esc_html_e( 'General', 'wcoct' ); ?></h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="wcoct_global_coupon"><?php esc_html_e( 'Global coupon code (Free)', 'wcoct' ); ?></label></th>
                        <td>
                            <input type="text" id="wcoct_global_coupon" name="wcoct_global_coupon" value="<?php echo esc_attr( $settings['global_coupon'] ); ?>" class="regular-text" />
                            <p class="description">
                                <?php esc_html_e( 'This coupon will be applied when the button is clicked. Leave empty (Pro) if using “Best coupon from list”.', 'wcoct' ); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wcoct_button_text_cart"><?php esc_html_e( 'Button text (cart)', 'wcoct' ); ?></label></th>
                        <td>
                            <input type="text" id="wcoct_button_text_cart" name="wcoct_button_text_cart" value="<?php echo esc_attr( $settings['button_text_cart'] ); ?>" class="regular-text" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wcoct_button_text_checkout"><?php esc_html_e( 'Button text (checkout)', 'wcoct' ); ?></label></th>
                        <td>
                            <input type="text" id="wcoct_button_text_checkout" name="wcoct_button_text_checkout" value="<?php echo esc_attr( $settings['button_text_checkout'] ); ?>" class="regular-text" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Display locations', 'wcoct' ); ?></th>
                        <td>
                            <label><input type="checkbox" name="wcoct_show_on_cart" value="1" <?php checked( $settings['show_on_cart'], 1 ); ?> /> <?php esc_html_e( 'Cart page', 'wcoct' ); ?></label><br/>
                            <label><input type="checkbox" name="wcoct_show_on_checkout" value="1" <?php checked( $settings['show_on_checkout'], 1 ); ?> /> <?php esc_html_e( 'Checkout page', 'wcoct' ); ?></label><br/>
                            <label><input type="checkbox" name="wcoct_show_on_mini_cart" value="1" <?php checked( $settings['show_on_mini_cart'], 1 ); ?> /> <?php esc_html_e( 'Mini cart (widget) (Pro)', 'wcoct' ); ?></label>
                            <?php if ( ! self::is_pro() ) : ?>
                                <p class="description"><em><?php esc_html_e( 'Mini-cart placement requires an active Pro license.', 'wcoct' ); ?></em></p>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wcoct_min_cart_total"><?php esc_html_e( 'Minimum cart total (Pro targeting)', 'wcoct' ); ?></label></th>
                        <td>
                            <input type="number" step="0.01" id="wcoct_min_cart_total" name="wcoct_min_cart_total" value="<?php echo esc_attr( $settings['min_cart_total'] ); ?>" />
                            <p class="description">
                                <?php esc_html_e( 'Show the button only when cart total is at least this amount. Requires Pro.', 'wcoct' ); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Limit by user role (Pro targeting)', 'wcoct' ); ?></th>
                        <td>
                            <?php foreach ( $all_roles as $role_key => $role ) : ?>
                                <label>
                                    <input type="checkbox" name="wcoct_roles_limit[]" value="<?php echo esc_attr( $role_key ); ?>" <?php checked( in_array( $role_key, $settings['roles_limit'], true ), true ); ?> />
                                    <?php echo esc_html( $role['name'] ); ?>
                                </label><br/>
                            <?php endforeach; ?>
                            <p class="description">
                                <?php esc_html_e( 'If any roles are selected, the button is shown only for those roles. Requires Pro.', 'wcoct' ); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <hr/>

                <h2><?php esc_html_e( 'Pro – Smart Coupon Logic', 'wcoct' ); ?></h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Best coupon from list', 'wcoct' ); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wcoct_enable_auto_best" value="1" <?php checked( $settings['enable_auto_best'], 1 ); ?> />
                                <?php esc_html_e( 'Automatically pick the best coupon from the list below.', 'wcoct' ); ?>
                            </label>
                            <?php if ( ! self::is_pro() ) : ?>
                                <p class="description"><em><?php esc_html_e( 'Requires an active Pro license. In Free version, only the global coupon is used.', 'wcoct' ); ?></em></p>
                            <?php else : ?>
                                <p class="description"><?php esc_html_e( 'The plugin will evaluate each coupon against the current cart and choose the highest estimated discount.', 'wcoct' ); ?></p>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wcoct_candidate_coupons"><?php esc_html_e( 'Candidate coupons (comma-separated)', 'wcoct' ); ?></label></th>
                        <td>
                            <textarea id="wcoct_candidate_coupons" name="wcoct_candidate_coupons" rows="3" cols="60"><?php echo esc_textarea( $settings['candidate_coupons'] ); ?></textarea>
                            <p class="description"><?php esc_html_e( 'Example: SPRING10, VIP20, FREESHIP', 'wcoct' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Show savings preview (Pro)', 'wcoct' ); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wcoct_show_savings_preview" value="1" <?php checked( $settings['show_savings_preview'], 1 ); ?> />
                                <?php esc_html_e( 'Show a “Save approx €X” preview based on the selected coupon.', 'wcoct' ); ?>
                            </label>
                            <?php if ( ! self::is_pro() ) : ?>
                                <p class="description"><em><?php esc_html_e( 'Requires an active Pro license.', 'wcoct' ); ?></em></p>
                            <?php endif; ?>
                        </td>
                    </tr>
                </table>

                <hr/>

                <h2><?php esc_html_e( 'License', 'wcoct' ); ?></h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Current status', 'wcoct' ); ?></th>
                        <td>
                            <?php
                            if ( $status === 'valid' ) {
                                echo '<span style="color:green;font-weight:bold;">' . esc_html__( 'Active', 'wcoct' ) . '</span>';
                            } elseif ( $status === 'grace' ) {
                                echo '<span style="color:#e6a700;font-weight:bold;">' . esc_html__( 'Grace period', 'wcoct' ) . '</span>';
                            } else {
                                echo '<span style="color:#cc0000;font-weight:bold;">' . esc_html__( 'Inactive', 'wcoct' ) . '</span>';
                            }
                            ?>
                            <p class="description">
                                <?php
                                printf(
                                    esc_html__( 'License is bound to domain: %s', 'wcoct' ),
                                    '<code>' . esc_html( $domain ) . '</code>'
                                );
                                ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wcoct_license_key"><?php esc_html_e( 'License key', 'wcoct' ); ?></label></th>
                        <td>
                            <input type="text" id="wcoct_license_key" name="wcoct_license_key" value="<?php echo esc_attr( $opt['key'] ); ?>" class="regular-text" />
                            <p class="description"><?php esc_html_e( 'Enter the license key you created / received via the license server.', 'wcoct' ); ?></p>
                        </td>
                    </tr>
                </table>

                <?php submit_button( __( 'Save settings', 'wcoct' ) ); ?>

            </form>
        </div>
        <?php
    }

    /**
     * Call remote Sparkcut license server.
     *
     * @param string $key
     * @param string $action activate|validate
     * @return array
     */
    private function remote_request( $key, $action ) {
        $body = [
            'action'      => $action,
            'license_key' => $key,
            'product'     => WCOCT_PRODUCT_SLUG,
            'domain'      => $this->domain(),
        ];

        $response = wp_remote_post(
            WCOCT_LICENSE_API,
            [
                'timeout' => 10,
                'body'    => $body,
            ]
        );

        if ( is_wp_error( $response ) ) {
            return [
                'ok'     => false,
                'msg'    => $response->get_error_message(),
                'active' => false,
            ];
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );

        if ( 200 !== $code ) {
            return [
                'ok'     => false,
                'msg'    => 'HTTP ' . $code,
                'active' => false,
            ];
        }

        $data = json_decode( $body, true );
        if ( ! is_array( $data ) ) {
            return [
                'ok'     => false,
                'msg'    => 'Invalid JSON from license server',
                'active' => false,
            ];
        }

        $data = wp_parse_args(
            $data,
            [
                'ok'     => false,
                'msg'    => '',
                'active' => true,
            ]
        );

        return $data;
    }

    /**
     * Persist validation/activation result and manage grace.
     *
     * @param array       $res
     * @param array       $opt
     * @param bool        $show_admin_notice
     * @param null|string $new_key
     */
    private function handle_validation_result( $res, $opt, $show_admin_notice = false, $new_key = null ) {
        $now = time();

        if ( null !== $new_key ) {
            $opt['key'] = $new_key;
        }

        if ( ! empty( $res['ok'] ) && ! empty( $res['active'] ) ) {
            $opt['activated']   = true;
            $opt['grace_until'] = 0;
            $opt['last_check']  = $now;

            update_option( self::OPT_KEY, $opt );
            set_transient( self::TRANSIENT_STATE, 'valid', self::CHECK_INTERVAL );

            if ( $show_admin_notice ) {
                $msg = $res['msg'] ? $res['msg'] : __( 'License validated.', 'wcoct' );
                add_action(
                    'admin_notices',
                    static function () use ( $msg ) {
                        echo '<div class="notice notice-success"><p>' . esc_html( $msg ) . '</p></div>';
                    }
                );
            }

            return;
        }

        // Failed or inactive: set / extend grace.
        $opt['activated']  = false;
        $opt['last_check'] = $now;

        if ( empty( $opt['grace_until'] ) || $opt['grace_until'] < $now ) {
            $opt['grace_until'] = $now + ( self::GRACE_HOURS * HOUR_IN_SECONDS );
        }

        update_option( self::OPT_KEY, $opt );

        if ( $opt['grace_until'] > $now ) {
            set_transient( self::TRANSIENT_STATE, 'grace', self::CHECK_INTERVAL );
        } else {
            set_transient( self::TRANSIENT_STATE, 'invalid', self::CHECK_INTERVAL );
        }

        if ( $show_admin_notice ) {
            $msg = ! empty( $res['msg'] ) ? $res['msg'] : __( 'License validation failed.', 'wcoct' );
            add_action(
                'admin_notices',
                static function () use ( $msg ) {
                    echo '<div class="notice notice-warning"><p>' . esc_html( $msg ) . '</p></div>';
                }
            );
        }
    }

    /**
     * Domain to bind license to.
     *
     * @return string
     */
    private function domain() {
        $home = home_url();
        $host = wp_parse_url( $home, PHP_URL_HOST );
        return $host ? $host : $home;
    }
}

/**
 * Main One-Click Coupon logic.
 *
 * Free:
 *  - One global coupon.
 *  - Button on cart + checkout.
 *
 * Pro:
 *  - Best coupon from a list, based on estimated discount.
 *  - Savings preview.
 *  - Mini-cart placement.
 *  - Targeting: minimum cart total + user roles.
 */
class WCOCT_Plugin {

    /**
     * Bootstrap hooks.
     */
    public function init() {
        if ( ! class_exists( 'WooCommerce' ) ) {
            return;
        }

        // Handle "apply coupon" requests before output.
        add_action( 'init', [ $this, 'maybe_apply_coupon_request' ] );

        // Frontend button placements.
        add_action( 'wp', [ $this, 'register_frontend_hooks' ] );
    }

    /**
     * Register button placements based on settings.
     */
    public function register_frontend_hooks() {
        $settings = $this->get_settings();

        if ( ! WC()->cart ) {
            return;
        }

        // Cart button.
        if ( ! empty( $settings['show_on_cart'] ) ) {
            add_action( 'woocommerce_cart_totals_after_order_total', [ $this, 'output_button_cart' ] );
        }

        // Checkout button.
        if ( ! empty( $settings['show_on_checkout'] ) ) {
            add_action( 'woocommerce_review_order_after_order_total', [ $this, 'output_button_checkout' ] );
        }

        // Mini-cart (Pro only).
        if ( WCOCT_License::is_pro() && ! empty( $settings['show_on_mini_cart'] ) ) {
            add_action( 'woocommerce_widget_shopping_cart_before_buttons', [ $this, 'output_button_mini_cart' ] );
        }
    }

    /**
     * Get settings with defaults.
     *
     * @return array
     */
    private function get_settings() {
        $defaults = [
            'global_coupon'        => '',
            'button_text_cart'     => __( 'Apply discount', 'wcoct' ),
            'button_text_checkout' => __( 'Apply discount', 'wcoct' ),
            'show_on_cart'         => 1,
            'show_on_checkout'     => 1,
            'show_on_mini_cart'    => 0,
            'min_cart_total'       => 0.0,
            'roles_limit'          => [],
            'enable_auto_best'     => 0,
            'candidate_coupons'    => '',
            'show_savings_preview' => 0,
        ];
        $settings = get_option( 'wcoct_settings', [] );
        return wp_parse_args( $settings, $defaults );
    }

    /**
     * Decide if we should display the button to this user / cart.
     *
     * @return bool
     */
    private function can_show_button() {
        $settings = $this->get_settings();

        if ( ! WC()->cart ) {
            return false;
        }

        // Cart total targeting (Pro).
        if ( WCOCT_License::is_pro() && $settings['min_cart_total'] > 0 ) {
            $total = WC()->cart->get_subtotal();
            if ( $total < $settings['min_cart_total'] ) {
                return false;
            }
        }

        // Role targeting (Pro).
        if ( WCOCT_License::is_pro() && ! empty( $settings['roles_limit'] ) ) {
            if ( is_user_logged_in() ) {
                $user     = wp_get_current_user();
                $user_roles = (array) $user->roles;
                $match      = array_intersect( $user_roles, $settings['roles_limit'] );
                if ( empty( $match ) ) {
                    return false;
                }
            } else {
                // Guest has no roles; if roles_limit is set, hide.
                return false;
            }
        }

        return true;
    }

    /**
     * Output button on cart page.
     */
    public function output_button_cart() {
        if ( ! $this->can_show_button() ) {
            return;
        }
        $settings = $this->get_settings();
        $this->render_button( 'cart', $settings['button_text_cart'] );
    }

    /**
     * Output button on checkout.
     */
    public function output_button_checkout() {
        if ( ! $this->can_show_button() ) {
            return;
        }
        $settings = $this->get_settings();
        $this->render_button( 'checkout', $settings['button_text_checkout'] );
    }

    /**
     * Output button in mini-cart (Pro).
     */
    public function output_button_mini_cart() {
        if ( ! WCOCT_License::is_pro() ) {
            return;
        }
        if ( ! $this->can_show_button() ) {
            return;
        }
        $settings = $this->get_settings();
        $this->render_button( 'mini_cart', $settings['button_text_cart'] );
    }

    /**
     * Render shared button HTML.
     *
     * @param string $context cart|checkout|mini_cart
     * @param string $text
     */
    private function render_button( $context, $text ) {
        $apply_url = add_query_arg(
            [
                'wcoct_apply' => 1,
                'wcoct_ctx'   => $context,
            ],
            wc_get_cart_url()
        );

        $settings = $this->get_settings();

        // Savings preview (Pro only).
        $preview_html = '';
        if ( WCOCT_License::is_pro() && ! empty( $settings['show_savings_preview'] ) ) {
            $estimate = $this->estimate_best_coupon_savings();
            if ( $estimate > 0 ) {
                $currency      = get_woocommerce_currency_symbol();
                $formatted_amt = wc_price( $estimate );
                $preview_html  = '<div class="wcoct-savings-preview" style="margin-top:4px;font-size:0.9em;color:#008000;">' .
                    sprintf(
                        /* translators: %s: formatted amount */
                        esc_html__( 'Save approximately %s with the best coupon.', 'wcoct' ),
                        wp_kses_post( $formatted_amt )
                    ) .
                    '</div>';
            }
        }

        echo '<div class="wcoct-wrapper" style="margin-top:10px;text-align:left;">';
        echo '<a href="' . esc_url( $apply_url ) . '" class="button wcoct-apply-button" style="background:#0073aa;border-color:#0073aa;color:#fff;">' . esc_html( $text ) . '</a>';
        echo $preview_html; // already escaped.
        echo '</div>';
    }

    /**
     * Handle apply coupon request triggered by button click.
     */
    public function maybe_apply_coupon_request() {
        if ( empty( $_GET['wcoct_apply'] ) ) {
            return;
        }

        if ( ! WC()->cart ) {
            return;
        }

        // Basic nonce would be ideal; but query param is sufficient here since this is idempotent.
        $settings = $this->get_settings();
        $coupon   = $this->determine_coupon_code( $settings );

        if ( ! $coupon ) {
            wc_add_notice( __( 'No available coupon could be applied at this time.', 'wcoct' ), 'notice' );
            wp_safe_redirect( wc_get_cart_url() );
            exit;
        }

        $coupon_code = wc_format_coupon_code( $coupon );

        if ( WC()->cart->has_discount( $coupon_code ) ) {
            wc_add_notice( __( 'Coupon is already applied.', 'wcoct' ), 'notice' );
            wp_safe_redirect( wc_get_cart_url() );
            exit;
        }

        $result = WC()->cart->apply_coupon( $coupon_code );
        if ( is_wp_error( $result ) ) {
            wc_add_notice( $result->get_error_message(), 'error' );
        } elseif ( ! $result ) {
            wc_add_notice( __( 'Coupon could not be applied.', 'wcoct' ), 'error' );
        } else {
            wc_add_notice( __( 'Coupon applied successfully.', 'wcoct' ), 'success' );
        }

        wp_safe_redirect( wc_get_cart_url() );
        exit;
    }

    /**
     * Decide which coupon to apply (Free vs Pro logic).
     *
     * @param array $settings
     * @return string|null
     */
    private function determine_coupon_code( $settings ) {
        // Pro: best coupon from list?
        if ( WCOCT_License::is_pro() && ! empty( $settings['enable_auto_best'] ) && ! empty( $settings['candidate_coupons'] ) ) {
            $best = $this->pick_best_coupon_from_list( $settings['candidate_coupons'] );
            if ( $best ) {
                return $best;
            }
        }

        // Fallback: global coupon (Free + Pro).
        if ( ! empty( $settings['global_coupon'] ) ) {
            return $settings['global_coupon'];
        }

        return null;
    }

    /**
     * Estimate savings of best coupon for preview.
     *
     * @return float
     */
    private function estimate_best_coupon_savings() {
        $settings = $this->get_settings();

        if ( ! WCOCT_License::is_pro() ) {
            return 0.0;
        }
        if ( empty( $settings['enable_auto_best'] ) || empty( $settings['candidate_coupons'] ) ) {
            return 0.0;
        }
        if ( ! WC()->cart ) {
            return 0.0;
        }

        $subtotal = WC()->cart->get_subtotal();
        if ( $subtotal <= 0 ) {
            return 0.0;
        }

        $best_discount = 0.0;

        $codes = array_filter(
            array_map(
                'trim',
                explode( ',', $settings['candidate_coupons'] )
            )
        );

        foreach ( $codes as $code ) {
            $code   = wc_format_coupon_code( $code );
            $coupon = new WC_Coupon( $code );
            if ( ! $coupon || ! $coupon->get_id() ) {
                continue;
            }
            if ( ! $coupon->is_valid() ) {
                continue;
            }

            $amount        = floatval( $coupon->get_amount() );
            $discount_type = $coupon->get_discount_type();
            $estimated     = 0.0;

            if ( in_array( $discount_type, [ 'percent', 'percent_cart' ], true ) ) {
                $estimated = $subtotal * ( $amount / 100.0 );
            } elseif ( in_array( $discount_type, [ 'fixed_cart', 'fixed_product' ], true ) ) {
                $estimated = $amount;
            } else {
                $estimated = $amount;
            }

            if ( $estimated > $best_discount ) {
                $best_discount = $estimated;
            }
        }

        return max( 0.0, $best_discount );
    }

    /**
     * Return the "best" coupon from a comma-separated list based on estimated discount.
     *
     * @param string $list
     * @return string|null
     */
    private function pick_best_coupon_from_list( $list ) {
        if ( ! WC()->cart ) {
            return null;
        }

        $subtotal = WC()->cart->get_subtotal();
        if ( $subtotal <= 0 ) {
            return null;
        }

        $codes = array_filter(
            array_map(
                'trim',
                explode( ',', $list )
            )
        );

        $best_code     = null;
        $best_discount = 0.0;

        foreach ( $codes as $code ) {
            $code   = wc_format_coupon_code( $code );
            $coupon = new WC_Coupon( $code );

            if ( ! $coupon || ! $coupon->get_id() ) {
                continue;
            }
            if ( ! $coupon->is_valid() ) {
                continue;
            }

            $amount        = floatval( $coupon->get_amount() );
            $discount_type = $coupon->get_discount_type();
            $estimated     = 0.0;

            if ( in_array( $discount_type, [ 'percent', 'percent_cart' ], true ) ) {
                $estimated = $subtotal * ( $amount / 100.0 );
            } elseif ( in_array( $discount_type, [ 'fixed_cart', 'fixed_product' ], true ) ) {
                $estimated = $amount;
            } else {
                $estimated = $amount;
            }

            if ( $estimated > $best_discount ) {
                $best_discount = $estimated;
                $best_code     = $code;
            }
        }

        return $best_code;
    }
}

// Bootstrap: license + plugin.
add_action(
    'plugins_loaded',
    static function () {
        if ( ! function_exists( 'wc' ) ) {
            return;
        }

        $lic = new WCOCT_License();
        $lic->init();

        $core = new WCOCT_Plugin();
        $core->init();
    }
);
