<?php
/**
 * Plugin Name: WooCommerce Stock Audit Log
 * Description: Logs stock movements (sales, refunds, manual edits, OpenPOS) and can retroactively rebuild history from orders & sales (Pro).
 * Version: 1.2.1
 * Author: Sparkcut Labs
 * Author URI: https://sparkcutlabs.com
 * Text Domain: wc-stock-audit-log
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'WCSAL_VERSION', '1.2.1' );
define( 'WCSAL_PLUGIN_FILE', __FILE__ );
define( 'WCSAL_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'WCSAL_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

// Option name used to store a per‑site license key.  When present this
// option enables the Pro features without requiring a central suite
// licensing manager.  Site owners can supply their key via the admin
// interface.  See WCSAL_Licensing::is_pro_active() for usage.
define( 'WCSAL_LICENSE_OPTION', 'wcsal_license_key' );

/**
 * Licensing wrapper compatible with your suite.
 *
 * This class DOES NOT hardcode any secrets or Stripe keys.
 * It defers to your central licensing logic via:
 *  - Filters:   es_plugin_is_pro( $is_pro, $slug )
 *  - Function:  es_is_plugin_pro( $slug )
 *  - Manager:   ES_Licensing_Manager::instance()->is_pro( $slug )
 */
class WCSAL_Licensing {

    const SLUG = 'wc_stock_audit_log';

    /**
     * Check if Pro is active for this plugin.
     */
    public static function is_pro_active() {
        // 1) Let central suite filter decide (highest priority).
        $filtered = apply_filters( 'es_plugin_is_pro', null, self::SLUG );
        if ( null !== $filtered ) {
            return (bool) $filtered;
        }

        // 2) Global helper function (if your suite exposes one).
        if ( function_exists( 'es_is_plugin_pro' ) ) {
            $res = es_is_plugin_pro( self::SLUG );
            return (bool) apply_filters( 'wcsal_is_pro_active', $res, self::SLUG );
        }

        // 3) Licensing manager class (if used in your suite).
        if ( class_exists( 'ES_Licensing_Manager' ) && method_exists( 'ES_Licensing_Manager', 'instance' ) ) {
            $manager = ES_Licensing_Manager::instance();
            if ( method_exists( $manager, 'is_pro' ) ) {
                $res = $manager->is_pro( self::SLUG );
                return (bool) apply_filters( 'wcsal_is_pro_active', $res, self::SLUG );
            }
        }

        // 4) Default: not Pro.
        // 4) Check for plugin‑specific license key.  If a license key has been
        //    entered in the plugin settings then treat this plugin as pro.
        $license_key = get_option( WCSAL_LICENSE_OPTION, '' );
        if ( ! empty( $license_key ) ) {
            return (bool) apply_filters( 'wcsal_is_pro_active', true, self::SLUG );
        }

        return (bool) apply_filters( 'wcsal_is_pro_active', false, self::SLUG );
    }
}

// ---------------------------------------------------------
//  Activation: create/update DB table
// ---------------------------------------------------------
register_activation_hook( __FILE__, 'wcsal_activate_plugin' );

function wcsal_activate_plugin() {
    global $wpdb;

    $table_name      = $wpdb->prefix . 'wc_stock_audit_log';
    $charset_collate = $wpdb->get_charset_collate();

    // user_id added for tracking who performed the action (customer or staff).
    $sql = "CREATE TABLE {$table_name} (
        id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
        product_id BIGINT UNSIGNED NOT NULL,
        variation_id BIGINT UNSIGNED DEFAULT 0,
        warehouse_id BIGINT UNSIGNED DEFAULT 0,
        user_id BIGINT UNSIGNED DEFAULT 0,
        qty_change DECIMAL(20,8) NOT NULL,
        qty_before DECIMAL(20,8) DEFAULT NULL,
        qty_after DECIMAL(20,8) DEFAULT NULL,
        event_type VARCHAR(50) NOT NULL,
        event_source VARCHAR(50) NOT NULL,
        source_id BIGINT UNSIGNED DEFAULT NULL,
        note TEXT NULL,
        created_at DATETIME NOT NULL,
        PRIMARY KEY  (id),
        KEY product_id (product_id),
        KEY warehouse_id (warehouse_id),
        KEY user_id (user_id),
        KEY event_type (event_type),
        KEY created_at (created_at)
    ) {$charset_collate};";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta( $sql );
}

// ---------------------------------------------------------
//  Bootstrap
// ---------------------------------------------------------
add_action( 'plugins_loaded', 'wcsal_init_plugin', 11 );

function wcsal_init_plugin() {
    if ( ! class_exists( 'WooCommerce' ) ) {
        return;
    }

    load_plugin_textdomain( 'wc-stock-audit-log', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );

    WCSAL_Logger::instance();
    WCSAL_Admin::instance();

    if ( WCSAL_Licensing::is_pro_active() ) {
        WCSAL_Rebuilder::instance();
    }
}

// ---------------------------------------------------------
//  Core Logger Class
// ---------------------------------------------------------
class WCSAL_Logger {

    private static $instance = null;

    public static function instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        // WooCommerce order-based reductions (sales).
        add_action( 'woocommerce_reduce_order_stock', [ $this, 'log_order_stock_reduction' ], 10, 1 );

        // Manual stock changes via product save.
        add_action( 'woocommerce_product_set_stock', [ $this, 'log_manual_stock_change' ], 10, 1 );

        // Refunds.
        add_action( 'woocommerce_order_refunded', [ $this, 'log_order_refund' ], 10, 2 );

        // Example OpenPOS hook (you can replace with actual OpenPOS hooks).
        add_action( 'wcsal_openpos_stock_change', [ $this, 'log_openpos_stock_change' ], 10, 6 );
    }

    /**
     * Generic insert helper.
     */
    public static function insert_log( $args ) {
        global $wpdb;
        $table = $wpdb->prefix . 'wc_stock_audit_log';

        $defaults = [
            'product_id'   => 0,
            'variation_id' => 0,
            'warehouse_id' => 0,
            'user_id'      => 0,
            'qty_change'   => 0,
            'qty_before'   => null,
            'qty_after'    => null,
            'event_type'   => '',
            'event_source' => '',
            'source_id'    => null,
            'note'         => '',
            'created_at'   => current_time( 'mysql' ),
        ];

        $data = wp_parse_args( $args, $defaults );

        $wpdb->insert(
            $table,
            $data,
            [
                '%d', // product_id
                '%d', // variation_id
                '%d', // warehouse_id
                '%d', // user_id
                '%f', // qty_change
                '%f', // qty_before
                '%f', // qty_after
                '%s', // event_type
                '%s', // event_source
                '%d', // source_id
                '%s', // note
                '%s', // created_at
            ]
        );
    }

    /**
     * Infer warehouse from order / metadata.
     * Adjust to your OpenPOS storage model if needed.
     */
    public static function get_warehouse_from_order_item( $order, $item ) {
        // Example: OpenPOS may store warehouse id in meta like _op_warehouse_id
        $warehouse_id = (int) $item->get_meta( '_op_warehouse_id', true );

        if ( ! $warehouse_id ) {
            $warehouse_id = (int) $order->get_meta( '_op_warehouse_id', true );
        }

        return (int) apply_filters( 'wcsal_get_warehouse_from_order_item', $warehouse_id, $order, $item );
    }

    // ---------------------------------------------------------
    //  WooCommerce: Sales
    // ---------------------------------------------------------
    public function log_order_stock_reduction( $order ) {
        // Hook for order stock reduction.  Normally triggered for completed/processing orders.
        // For our plugin we also treat custom "credit" orders as refunds rather than sales.
        // A credit order increases stock, so we flip the sign of qty_change and use a refund event type.
        if ( ! $order instanceof WC_Order ) {
            $order = wc_get_order( $order );
        }
        if ( ! $order ) {
            return;
        }

        $user_id   = (int) $order->get_user_id();
        $user_obj  = $user_id ? get_user_by( 'id', $user_id ) : null;
        $user_name = $user_obj ? $user_obj->display_name : '';

        $order_status = $order->get_status();

        foreach ( $order->get_items() as $item_id => $item ) {
            if ( ! $item instanceof WC_Order_Item_Product ) {
                continue;
            }

            $product      = $item->get_product();
            $product_id   = $item->get_product_id();
            $variation_id = $item->get_variation_id();
            $qty          = $item->get_quantity();
            $warehouse_id = self::get_warehouse_from_order_item( $order, $item );

            if ( ! $product ) {
                continue;
            }

            $is_credit    = ( 'credit' === $order_status );
            $qty_change   = $is_credit ? abs( $qty ) : -1 * $qty;
            $event_type   = $is_credit ? 'refund' : 'sale';
            $event_source = $is_credit ? 'woocommerce_credit' : 'woocommerce_order';

            $stock_before = wc_get_stock_quantity( $variation_id ?: $product_id );
            if ( null === $stock_before ) {
                $stock_after = null;
            } else {
                $stock_after = $is_credit ? $stock_before + $qty_change : $stock_before - $qty;
            }

            $order_link = admin_url( 'post.php?post=' . $order->get_id() . '&action=edit' );
            $note_parts = [];
            $note_parts[] = sprintf( '<a href="%s">Order #%d</a>', esc_url( $order_link ), $order->get_id() );
            $note_parts[] = sprintf( 'item #%d', $item_id );
            if ( $user_name ) {
                $note_parts[] = sprintf( 'by %s', esc_html( $user_name ) );
            }
            $note = implode( ' ', $note_parts );

            self::insert_log(
                [
                    'product_id'   => $product_id,
                    'variation_id' => $variation_id,
                    'warehouse_id' => $warehouse_id,
                    'user_id'      => $user_id,
                    'qty_change'   => $qty_change,
                    'qty_before'   => $stock_before,
                    'qty_after'    => $stock_after,
                    'event_type'   => $event_type,
                    'event_source' => $event_source,
                    'source_id'    => $order->get_id(),
                    'note'         => $note,
                ]
            );
        }
    }

    // ---------------------------------------------------------
    //  WooCommerce: Manual stock edits
    // ---------------------------------------------------------
    public function log_manual_stock_change( $product ) {
        if ( ! $product instanceof WC_Product ) {
            return;
        }

        $product_id   = $product->get_id();
        $variation_id = $product->is_type( 'variation' ) ? $product_id : 0;
        $new_stock    = $product->get_stock_quantity();

        $prev_stock  = $this->get_last_stock_for_product( $product_id, $variation_id );
        // Calculate change.  If there is no previous log, treat previous stock as zero.
        if ( null === $prev_stock && null !== $new_stock ) {
            $qty_change = (float) $new_stock;
        } elseif ( null !== $prev_stock && null !== $new_stock ) {
            $qty_change = (float) ( $new_stock - $prev_stock );
        } else {
            $qty_change = 0.0;
        }

        // Skip logging if nothing has changed.
        if ( abs( $qty_change ) < 0.000001 ) {
            return;
        }

        $user_id = get_current_user_id();

        self::insert_log(
            [
                'product_id'   => $product_id,
                'variation_id' => $variation_id,
                'warehouse_id' => 0,
                'user_id'      => $user_id,
                'qty_change'   => $qty_change,
                // If no previous stock, record null so UI shows a dash rather than assuming zero.
                'qty_before'   => ( null === $prev_stock ? null : $prev_stock ),
                'qty_after'    => $new_stock,
                'event_type'   => 'manual_adjustment',
                'event_source' => 'product_edit',
                'source_id'    => null,
                'note'         => 'Manual stock edit in product page',
            ]
        );
    }

    private function get_last_stock_for_product( $product_id, $variation_id = 0 ) {
        global $wpdb;
        $table = $wpdb->prefix . 'wc_stock_audit_log';

        $sql = $wpdb->prepare(
            "SELECT qty_after FROM {$table}
             WHERE product_id = %d AND variation_id = %d
             ORDER BY created_at DESC, id DESC
             LIMIT 1",
            $product_id,
            $variation_id
        );

        return $wpdb->get_var( $sql );
    }

    // ---------------------------------------------------------
    //  WooCommerce: Refunds
    // ---------------------------------------------------------
    public function log_order_refund( $order_id, $refund_id ) {
        $order  = wc_get_order( $order_id );
        $refund = wc_get_order( $refund_id );

        if ( ! $order || ! $refund ) {
            return;
        }

        $user_id = (int) $order->get_user_id();

        foreach ( $refund->get_items() as $item_id => $item ) {
            if ( ! $item instanceof WC_Order_Item_Product ) {
                continue;
            }

            $product       = $item->get_product();
            $product_id    = $item->get_product_id();
            $variation_id  = $item->get_variation_id();
            $qty           = $item->get_quantity(); // Usually negative.
            $warehouse_id  = self::get_warehouse_from_order_item( $order, $item );

            if ( ! $product ) {
                continue;
            }

            $qty_change  = abs( $qty ); // Refund returns stock.
            $stock_before = wc_get_stock_quantity( $variation_id ?: $product_id );
            $stock_after  = is_null( $stock_before ) ? null : $stock_before + $qty_change;

            // Build note with link to the order, refund, item ID and user name.
            $order_link = admin_url( 'post.php?post=' . $order_id . '&action=edit' );
            $note_parts = [];
            $note_parts[] = sprintf( '<a href="%s">Order #%d</a>', esc_url( $order_link ), $order_id );
            $note_parts[] = sprintf( 'refund #%d', $refund_id );
            $note_parts[] = sprintf( 'item #%d', $item_id );
            $user_obj  = $user_id ? get_user_by( 'id', $user_id ) : null;
            $user_name = $user_obj ? $user_obj->display_name : '';
            if ( $user_name ) {
                $note_parts[] = sprintf( 'by %s', esc_html( $user_name ) );
            }
            $note = implode( ' ', $note_parts );

            self::insert_log(
                [
                    'product_id'   => $product_id,
                    'variation_id' => $variation_id,
                    'warehouse_id' => $warehouse_id,
                    'user_id'      => $user_id,
                    'qty_change'   => $qty_change,
                    'qty_before'   => $stock_before,
                    'qty_after'    => $stock_after,
                    'event_type'   => 'refund',
                    'event_source' => 'woocommerce_refund',
                    'source_id'    => $refund_id,
                    'note'         => $note,
                ]
            );
        }
    }

    // ---------------------------------------------------------
    //  OpenPOS: Generic hook (STUB)
// ---------------------------------------------------------
    /**
     * Example hook you can trigger from OpenPOS integration:
     *
     * do_action(
     *     'wcsal_openpos_stock_change',
     *     $product_id,
     *     $variation_id,
     *     $warehouse_id,
     *     $qty_change, // + for receiving / transfer in, - for sale / transfer out
     *     'OpenPOS Receiving #123',
     *     $user_id // operator user ID, optional
     * );
     */
    public function log_openpos_stock_change( $product_id, $variation_id, $warehouse_id, $qty_change, $note = '', $user_id = 0 ) {
        $product_id   = (int) $product_id;
        $variation_id = (int) $variation_id;
        $warehouse_id = (int) $warehouse_id;
        $qty_change   = (float) $qty_change;
        $user_id      = (int) $user_id;

        $stock_before = wc_get_stock_quantity( $variation_id ?: $product_id );
        $stock_after  = is_null( $stock_before ) ? null : $stock_before + $qty_change;

        self::insert_log(
            [
                'product_id'   => $product_id,
                'variation_id' => $variation_id,
                'warehouse_id' => $warehouse_id,
                'user_id'      => $user_id,
                'qty_change'   => $qty_change,
                'qty_before'   => $stock_before,
                'qty_after'    => $stock_after,
                'event_type'   => 'openpos',
                'event_source' => 'openpos',
                'source_id'    => null,
                'note'         => $note,
            ]
        );
    }
}

// ---------------------------------------------------------
//  Pro Rebuilder (Retrospective, batched, AJAX)
// ---------------------------------------------------------
class WCSAL_Rebuilder {

    private static $instance = null;

    const JOB_OPTION_KEY = 'wcsal_rebuild_job';

    private $batch_size = 100; // orders per AJAX step

    public static function instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action( 'admin_post_wcsal_rebuild_history', [ $this, 'start_rebuild_job' ] );
        add_action( 'wp_ajax_wcsal_rebuild_step', [ $this, 'ajax_rebuild_step' ] );
    }

    /**
     * Start a batched rebuild job from admin form.
     */
    public function start_rebuild_job() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( 'Not allowed.' );
        }

        check_admin_referer( 'wcsal_rebuild_history' );

        $from_date = ! empty( $_POST['wcsal_from_date'] ) ? sanitize_text_field( wp_unslash( $_POST['wcsal_from_date'] ) ) : '';
        $to_date   = ! empty( $_POST['wcsal_to_date'] ) ? sanitize_text_field( wp_unslash( $_POST['wcsal_to_date'] ) ) : '';

        // Get total matching orders efficiently.
        $date_query = [];
        if ( $from_date ) {
            $date_query[] = [
                'after'     => $from_date . ' 00:00:00',
                'inclusive' => true,
            ];
        }
        if ( $to_date ) {
            $date_query[] = [
                'before'    => $to_date . ' 23:59:59',
                'inclusive' => true,
            ];
        }

        // Include custom credit status so credit orders are rebuilt as refunds.
        $statuses = [ 'wc-processing', 'wc-completed', 'wc-refunded', 'wc-cancelled', 'wc-credit' ];

        $count_query = new WP_Query(
            [
                'post_type'      => 'shop_order',
                'post_status'    => $statuses,
                'date_query'     => $date_query,
                'fields'         => 'ids',
                'posts_per_page' => 1,
                'no_found_rows'  => false, // we want found_posts
            ]
        );

        $total_orders = (int) $count_query->found_posts;

        // Clear previous rebuild-related entries so a new rebuild is clean.
        $this->clear_previous_rebuild_entries();

        $job = [
            'id'             => uniqid( 'wcsal_job_', true ),
            'from_date'      => $from_date,
            'to_date'        => $to_date,
            'batch_size'     => apply_filters( 'wcsal_rebuild_batch_size', $this->batch_size ),
            'offset'         => 0,
            'total_orders'   => $total_orders,
            'processed'      => 0,
            'status'         => 'running',
            'started_at'     => current_time( 'mysql' ),
            'finished_at'    => null,
            'last_message'   => '',
        ];

        update_option( self::JOB_OPTION_KEY, $job, false );

        wp_redirect(
            add_query_arg(
                [
                    'page'         => 'wcsal_audit_log',
                    'wcsal_job'    => 'started',
                    'wcsal_msg'    => 'rebuild_started',
                ],
                admin_url( 'admin.php' )
            )
        );
        exit;
    }

    /**
     * Return the current job (array or null).
     */
    public static function get_current_job() {
        $job = get_option( self::JOB_OPTION_KEY );
        if ( ! is_array( $job ) || empty( $job['id'] ) ) {
            return null;
        }
        return $job;
    }

    /**
     * AJAX handler: process a single batch.
     */
    public function ajax_rebuild_step() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => 'Not allowed' ] );
        }

        check_ajax_referer( 'wcsal_rebuild_step', 'nonce' );

        $job = self::get_current_job();
        if ( ! $job || 'running' !== $job['status'] ) {
            wp_send_json_error( [ 'message' => 'No running job.' ] );
        }

        $batch_size = (int) $job['batch_size'];
        $offset     = (int) $job['offset'];

        $date_query = [];
        if ( ! empty( $job['from_date'] ) ) {
            $date_query[] = [
                'after'     => $job['from_date'] . ' 00:00:00',
                'inclusive' => true,
            ];
        }
        if ( ! empty( $job['to_date'] ) ) {
            $date_query[] = [
                'before'    => $job['to_date'] . ' 23:59:59',
                'inclusive' => true,
            ];
        }

        // Include custom credit status so credit orders are rebuilt as refunds.
        $statuses = [ 'wc-processing', 'wc-completed', 'wc-refunded', 'wc-cancelled', 'wc-credit' ];

        $q = new WP_Query(
            [
                'post_type'      => 'shop_order',
                'post_status'    => $statuses,
                'date_query'     => $date_query,
                'fields'         => 'ids',
                'posts_per_page' => $batch_size,
                'offset'         => $offset,
                'no_found_rows'  => true,
            ]
        );

        $orders = $q->posts;

        if ( empty( $orders ) ) {
            // No more orders -> finish job + detect mismatches.
            $job['status']       = 'finished';
            $job['finished_at']  = current_time( 'mysql' );
            $job['last_message'] = 'Rebuild finished; checking mismatches.';
            update_option( self::JOB_OPTION_KEY, $job, false );

            $this->detect_mismatches();

            wp_send_json_success(
                [
                    'done'      => true,
                    'processed' => $job['processed'],
                    'total'     => $job['total_orders'],
                    'message'   => __( 'Rebuild complete. Mismatch detection done.', 'wc-stock-audit-log' ),
                ]
            );
        }

        // Process this batch of orders.
        foreach ( $orders as $order_id ) {
            $order = wc_get_order( $order_id );
            if ( ! $order ) {
                continue;
            }

            $this->process_order_for_rebuild( $order );
            $job['processed']++;
        }

        $job['offset'] = $offset + count( $orders );
        $job['last_message'] = sprintf(
            /* translators: 1: processed, 2: total */
            __( 'Processed %1$d / %2$d orders...', 'wc-stock-audit-log' ),
            $job['processed'],
            $job['total_orders']
        );

        update_option( self::JOB_OPTION_KEY, $job, false );

        $percent = ( $job['total_orders'] > 0 )
            ? min( 100, round( ( $job['processed'] / $job['total_orders'] ) * 100 ) )
            : 0;

        wp_send_json_success(
            [
                'done'      => false,
                'processed' => $job['processed'],
                'total'     => $job['total_orders'],
                'percent'   => $percent,
                'message'   => $job['last_message'],
            ]
        );
    }

    /**
     * Clear previous rebuild-related entries so a new rebuild is clean.
     */
    private function clear_previous_rebuild_entries() {
        global $wpdb;
        $table = $wpdb->prefix . 'wc_stock_audit_log';

        // Remove entries sourced from rebuild / mismatch detector.
        $wpdb->query(
            "DELETE FROM {$table}
             WHERE event_source IN ('woocommerce_order_rebuild', 'woocommerce_refund_rebuild', 'mismatch_detector')"
        );
    }

    /**
     * Process a single order for rebuild: create sale + refund log entries.
     */
    private function process_order_for_rebuild( WC_Order $order ) {
        $created_at = $order->get_date_created() ? $order->get_date_created()->date( 'Y-m-d H:i:s' ) : current_time( 'mysql' );
        $user_id    = (int) $order->get_user_id();

        // Determine order status once to avoid multiple lookups.
        $order_status = $order->get_status();
        // Determine user display name once.
        $user_obj  = $user_id ? get_user_by( 'id', $user_id ) : null;
        $user_name = $user_obj ? $user_obj->display_name : '';

        foreach ( $order->get_items() as $item_id => $item ) {
            if ( ! $item instanceof WC_Order_Item_Product ) {
                continue;
            }

            $product = $item->get_product();
            if ( ! $product ) {
                continue;
            }

            $product_id   = $item->get_product_id();
            $variation_id = $item->get_variation_id();
            $qty          = (float) $item->get_quantity();
            $warehouse_id = WCSAL_Logger::get_warehouse_from_order_item( $order, $item );

            // For credit orders treat movements as refunds to increase stock.
            $is_credit = ( 'credit' === $order_status );
            $qty_change = $is_credit ? abs( $qty ) : -1 * $qty;
            $event_type = $is_credit ? 'refund' : 'sale';
            $event_source = $is_credit ? 'woocommerce_credit_rebuild' : 'woocommerce_order_rebuild';

            // Build note: include link to order, item and user name.
            $order_link = admin_url( 'post.php?post=' . $order->get_id() . '&action=edit' );
            $note_parts = [];
            $note_parts[] = sprintf( '<a href="%s">Order #%d</a>', esc_url( $order_link ), $order->get_id() );
            $note_parts[] = sprintf( 'item #%d', $item_id );
            if ( $user_name ) {
                $note_parts[] = sprintf( 'by %s', esc_html( $user_name ) );
            }
            // Prefix with Rebuild: for clarity.
            $note = 'Rebuild: ' . implode( ' ', $note_parts );

            WCSAL_Logger::insert_log(
                [
                    'product_id'   => $product_id,
                    'variation_id' => $variation_id,
                    'warehouse_id' => $warehouse_id,
                    'user_id'      => $user_id,
                    'qty_change'   => $qty_change,
                    'qty_before'   => null,
                    'qty_after'    => null,
                    'event_type'   => $event_type,
                    'event_source' => $event_source,
                    'source_id'    => $order->get_id(),
                    'note'         => $note,
                    'created_at'   => $created_at,
                ]
            );
        }

        foreach ( $order->get_refunds() as $refund ) {
            /** @var WC_Order_Refund $refund */
            $refund_created_at = $refund->get_date_created() ? $refund->get_date_created()->date( 'Y-m-d H:i:s' ) : current_time( 'mysql' );

            foreach ( $refund->get_items() as $item_id => $item ) {
                if ( ! $item instanceof WC_Order_Item_Product ) {
                    continue;
                }

                $product = $item->get_product();
                if ( ! $product ) {
                    continue;
                }

                $product_id   = $item->get_product_id();
                $variation_id = $item->get_variation_id();
                $qty          = (float) abs( $item->get_quantity() );
                $warehouse_id = WCSAL_Logger::get_warehouse_from_order_item( $order, $item );

                // Build note with link to order/refund, item and user name.
                $order_link = admin_url( 'post.php?post=' . $order->get_id() . '&action=edit' );
                $note_parts = [];
                $note_parts[] = sprintf( '<a href="%s">Order #%d</a>', esc_url( $order_link ), $order->get_id() );
                $note_parts[] = sprintf( 'refund #%d', $refund->get_id() );
                $note_parts[] = sprintf( 'item #%d', $item_id );
                $user_obj  = $user_id ? get_user_by( 'id', $user_id ) : null;
                $user_name = $user_obj ? $user_obj->display_name : '';
                if ( $user_name ) {
                    $note_parts[] = sprintf( 'by %s', esc_html( $user_name ) );
                }
                $note = 'Rebuild: ' . implode( ' ', $note_parts );

                WCSAL_Logger::insert_log(
                    [
                        'product_id'   => $product_id,
                        'variation_id' => $variation_id,
                        'warehouse_id' => $warehouse_id,
                        'user_id'      => $user_id,
                        'qty_change'   => $qty,
                        'qty_before'   => null,
                        'qty_after'    => null,
                        'event_type'   => 'refund',
                        'event_source' => 'woocommerce_refund_rebuild',
                        'source_id'    => $refund->get_id(),
                        'note'         => $note,
                        'created_at'   => $refund_created_at,
                    ]
                );
            }
        }
    }

    /**
     * Detect mismatches between reconstructed movements and current stock.
     * Logs "untracked_adjustment" entries when mismatch detected.
     */
    public function detect_mismatches() {
        global $wpdb;

        $table = $wpdb->prefix . 'wc_stock_audit_log';

        $rows = $wpdb->get_results(
            "SELECT product_id, variation_id, warehouse_id, SUM(qty_change) as total_movement
             FROM {$table}
             GROUP BY product_id, variation_id, warehouse_id"
        );

        if ( ! $rows ) {
            return;
        }

        foreach ( $rows as $row ) {
            $product_id   = (int) $row->product_id;
            $variation_id = (int) $row->variation_id;

            $product = wc_get_product( $variation_id ?: $product_id );
            if ( ! $product ) {
                continue;
            }

            $current_stock = $product->get_stock_quantity();
            if ( null === $current_stock ) {
                continue; // unmanaged stock
            }

            $expected_stock = (float) $row->total_movement; // assumes initial stock 0
            $delta          = $current_stock - $expected_stock;

            if ( abs( $delta ) > 0.0001 ) {
                WCSAL_Logger::insert_log(
                    [
                        'product_id'   => $product_id,
                        'variation_id' => $variation_id,
                        'warehouse_id' => (int) $row->warehouse_id,
                        'user_id'      => 0,
                        'qty_change'   => $delta,
                        'qty_before'   => $expected_stock,
                        'qty_after'    => $current_stock,
                        'event_type'   => 'untracked_adjustment',
                        'event_source' => 'mismatch_detector',
                        // Use the product ID as the source identifier for mismatch
                        // adjustments so that the Source ID column is not left blank.
                        'source_id'    => $product_id ?: null,
                        'note'         => sprintf(
                            'Detected mismatch vs reconstructed movements. Auto-adjustment %s%0.3f',
                            $delta > 0 ? '+' : '',
                            $delta
                        ),
                        'created_at'   => current_time( 'mysql' ),
                    ]
                );

                $this->send_mismatch_email( $product, $row->warehouse_id, $delta, $current_stock, $expected_stock );
            }
        }
    }

    private function send_mismatch_email( $product, $warehouse_id, $delta, $current_stock, $expected_stock ) {
        $enabled = (bool) get_option( 'wcsal_enable_mismatch_email', false );
        if ( ! $enabled ) {
            return;
        }

        $recipient = get_option( 'wcsal_mismatch_email_recipient', get_option( 'admin_email' ) );
        $subject   = sprintf( '[Stock Audit] Mismatch detected for %s', $product->get_name() );
        $body      = sprintf(
            "A stock mismatch was detected.\n\nProduct: %s (ID: %d)\nWarehouse: %d\nDelta: %s%0.3f\nExpected stock: %0.3f\nCurrent stock: %0.3f\n\nPlease review the audit log for details.",
            $product->get_name(),
            $product->get_id(),
            (int) $warehouse_id,
            $delta > 0 ? '+' : '',
            $delta,
            $expected_stock,
            $current_stock
        );

        wp_mail( $recipient, $subject, $body );
    }
}

// ---------------------------------------------------------
//  Admin UI (DataTables, autocomplete filters, CSV/Excel export)
// ---------------------------------------------------------
class WCSAL_Admin {

    private static $instance = null;

    public static function instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action( 'admin_menu', [ $this, 'register_menu' ] );
        add_action( 'admin_init', [ $this, 'register_settings' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_assets' ] );

        // AJAX endpoints for autocomplete.
        add_action( 'wp_ajax_wcsal_search_products', [ $this, 'ajax_search_products' ] );
        add_action( 'wp_ajax_wcsal_search_categories', [ $this, 'ajax_search_categories' ] );
        add_action( 'wp_ajax_wcsal_search_users', [ $this, 'ajax_search_users' ] );

        // AJAX endpoint for warehouse search (select2 multi-select).
        add_action( 'wp_ajax_wcsal_search_warehouses', [ $this, 'ajax_search_warehouses' ] );
    }

    public function register_menu() {
        add_submenu_page(
            'woocommerce',
            __( 'Stock Audit Log', 'wc-stock-audit-log' ),
            __( 'Stock Audit Log', 'wc-stock-audit-log' ),
            'manage_woocommerce',
            'wcsal_audit_log',
            [ $this, 'render_page' ]
        );
    }

    public function register_settings() {
        register_setting( 'wcsal_settings', 'wcsal_enable_mismatch_email', [ 'type' => 'boolean', 'default' => false ] );
        register_setting( 'wcsal_settings', 'wcsal_mismatch_email_recipient', [ 'type' => 'string', 'default' => get_option( 'admin_email' ) ] );

        // Register a setting to capture the licence key.  This field allows
        // site owners to enter their own license key if a central licence
        // manager is not available.  A blank value indicates no licence.
        register_setting( 'wcsal_settings', WCSAL_LICENSE_OPTION, [ 'type' => 'string', 'default' => '' ] );
    }

    public function enqueue_assets( $hook ) {
        if ( 'woocommerce_page_wcsal_audit_log' !== $hook ) {
            return;
        }

        // jQuery + jQuery UI Autocomplete (WordPress core).
        wp_enqueue_script( 'jquery' );
        wp_enqueue_script( 'jquery-ui-autocomplete' );

        // (For WordPress.org, you would bundle DataTables locally instead of CDN.)
        wp_enqueue_style(
            'wcsal-datatables',
            'https://cdn.datatables.net/1.13.8/css/jquery.dataTables.min.css',
            [],
            '1.13.8'
        );
        wp_enqueue_style(
            'wcsal-datatables-buttons',
            'https://cdn.datatables.net/buttons/2.4.2/css/buttons.dataTables.min.css',
            [ 'wcsal-datatables' ],
            '2.4.2'
        );

        wp_enqueue_script(
            'wcsal-datatables',
            'https://cdn.datatables.net/1.13.8/js/jquery.dataTables.min.js',
            [ 'jquery' ],
            '1.13.8',
            true
        );
        wp_enqueue_script(
            'wcsal-datatables-buttons',
            'https://cdn.datatables.net/buttons/2.4.2/js/dataTables.buttons.min.js',
            [ 'wcsal-datatables' ],
            '2.4.2',
            true
        );
        wp_enqueue_script(
            'wcsal-datatables-jszip',
            'https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js',
            [],
            '3.10.1',
            true
        );
        wp_enqueue_script(
            'wcsal-datatables-html5',
            'https://cdn.datatables.net/buttons/2.4.2/js/buttons.html5.min.js',
            [ 'wcsal-datatables-buttons', 'wcsal-datatables-jszip' ],
            '2.4.2',
            true
        );

        // Select2 CSS & JS for enhanced multi-select dropdowns.
        wp_enqueue_style(
            'wcsal-select2',
            'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css',
            [],
            '4.1.0-rc.0'
        );
        wp_enqueue_script(
            'wcsal-select2',
            'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js',
            [ 'jquery' ],
            '4.1.0-rc.0',
            true
        );

        // Simple base styling for nicer UI.
        $css = '
        #wcsal-filters form {
            display:flex;
            flex-wrap:wrap;
            gap:12px;
            align-items:flex-end;
            margin-bottom:15px;
        }
        #wcsal-filters .wcsal-filter-group {
            display:flex;
            flex-direction:column;
        }
        #wcsal-filters label {
            font-weight:600;
            margin-bottom:2px;
        }
        #wcsal-filters input[type="text"],
        #wcsal-filters input[type="number"] {
            min-width:200px;
        }
        #wcsal-log-table_wrapper .dataTables_length {
            margin-right:20px;
        }
        #wcsal-log-table_wrapper .dt-buttons .dt-button {
            margin-right:5px;
        }
        ';
        wp_add_inline_style( 'wcsal-datatables', $css );
    }

    public function render_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        $message = isset( $_GET['wcsal_msg'] ) ? sanitize_text_field( wp_unslash( $_GET['wcsal_msg'] ) ) : '';

        if ( 'rebuild_started' === $message ) {
            echo '<div class="notice notice-info"><p>' . esc_html__( 'Rebuild started. The process will run in the background; keep this page open until it finishes.', 'wc-stock-audit-log' ) . '</p></div>';
        }

        $this->maybe_export_csv();

        $view = isset( $_GET['wcsal_view'] ) ? sanitize_text_field( wp_unslash( $_GET['wcsal_view'] ) ) : '';

        echo '<div class="wrap wcsal-wrap">';
        echo '<h1 style="margin-bottom:10px;">' . esc_html__( 'Stock Audit Log', 'wc-stock-audit-log' ) . '</h1>';
        echo '<p style="max-width:760px;color:#555;">' . esc_html__( 'Track every stock movement across orders, refunds, manual changes, and POS activity. Use the filters below to drill down by product, category, warehouse, or user.', 'wc-stock-audit-log' ) . '</p>';

        if ( 'product' === $view && ! empty( $_GET['product_id'] ) ) {
            $this->render_product_timeline( (int) $_GET['product_id'] );
        } else {
            $this->render_filters();
            $this->render_table();
        }

        if ( WCSAL_Licensing::is_pro_active() && class_exists( 'WCSAL_Rebuilder' ) ) {
            $this->render_rebuild_box();
        } else {
            $this->render_pro_upsell();
        }

        $this->render_settings_box();

        echo '</div>';

        $this->render_js();
    }

    // ----------------- AJAX AUTOCOMPLETE -------------------

    public function ajax_search_products() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error();
        }

        check_ajax_referer( 'wcsal_search', 'nonce' );

        $term = isset( $_GET['term'] ) ? sanitize_text_field( wp_unslash( $_GET['term'] ) ) : '';
        if ( '' === $term ) {
            wp_send_json( [] );
        }

        $args = [
            'limit'   => 20,
            'status'  => 'publish',
            'orderby' => 'title',
            'order'   => 'ASC',
            's'       => $term,
        ];

        $products = wc_get_products( $args );
        $results  = [];

        foreach ( $products as $product ) {
            /** @var WC_Product $product */
            $results[] = [
                'id'    => $product->get_id(),
                'label' => sprintf( '%s (#%d)', $product->get_name(), $product->get_id() ),
            ];
        }

        wp_send_json( $results );
    }

    public function ajax_search_categories() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error();
        }

        check_ajax_referer( 'wcsal_search', 'nonce' );

        $term = isset( $_GET['term'] ) ? sanitize_text_field( wp_unslash( $_GET['term'] ) ) : '';
        if ( '' === $term ) {
            wp_send_json( [] );
        }

        $cats = get_terms(
            [
                'taxonomy'   => 'product_cat',
                'hide_empty' => false,
                'search'     => $term,
                'number'     => 20,
            ]
        );

        $results = [];
        if ( ! is_wp_error( $cats ) && $cats ) {
            foreach ( $cats as $cat ) {
                $results[] = [
                    'id'    => $cat->term_id,
                    'label' => sprintf( '%s (#%d)', $cat->name, $cat->term_id ),
                ];
            }
        }

        wp_send_json( $results );
    }

    public function ajax_search_users() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error();
        }

        check_ajax_referer( 'wcsal_search', 'nonce' );

        $term = isset( $_GET['term'] ) ? sanitize_text_field( wp_unslash( $_GET['term'] ) ) : '';
        if ( '' === $term ) {
            wp_send_json( [] );
        }

        $users = get_users(
            [
                'search'         => '*' . $term . '*',
                'number'         => 20,
                'search_columns' => [ 'user_login', 'user_nicename', 'user_email', 'display_name' ],
            ]
        );

        $results = [];
        foreach ( $users as $user ) {
            $results[] = [
                'id'    => $user->ID,
                'label' => sprintf( '%s (%s)', $user->display_name, $user->user_email ),
            ];
        }

        wp_send_json( $results );
    }

    /**
     * AJAX search for warehouses.  Attempts to find warehouse names via a custom filter
     * or falls back to searching post titles across all public post types.  Results are
     * returned as arrays with `id` and `label` keys for Select2.
     */
    public function ajax_search_warehouses() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error();
        }

        check_ajax_referer( 'wcsal_search', 'nonce' );

        $term = isset( $_GET['term'] ) ? sanitize_text_field( wp_unslash( $_GET['term'] ) ) : '';
        if ( '' === $term ) {
            wp_send_json( [] );
        }

        $results = [];

        /**
         * Allow third-party plugins to supply warehouse search results.
         * The filter should return an array of arrays with `id` and `label` keys.
         *
         * @param array  $results Existing results.
         * @param string $search  Search term.
         */
        $filtered = apply_filters( 'wcsal_search_warehouses', [], $term );
        if ( ! empty( $filtered ) && is_array( $filtered ) ) {
            foreach ( $filtered as $wh ) {
                if ( isset( $wh['id'], $wh['label'] ) ) {
                    $results[] = [
                        'id'    => $wh['id'],
                        'label' => $wh['label'],
                    ];
                }
            }
        } else {
            // Fallback: search published posts across all post types by title.
            $posts = get_posts(
                [
                    'post_type'      => 'any',
                    'post_status'    => 'publish',
                    's'              => $term,
                    'posts_per_page' => 20,
                ]
            );

            foreach ( $posts as $post ) {
                $results[] = [
                    'id'    => $post->ID,
                    'label' => sprintf( '%s (#%d)', $post->post_title, $post->ID ),
                ];
            }
        }

        wp_send_json( $results );
    }

    // ----------------- FILTERS & TABLE ----------------------

    private function render_filters() {
        // Collect selected filter values (arrays for multi-selects).
        $selected_product_ids   = ! empty( $_GET['wcsal_product_ids'] ) ? array_map( 'intval', (array) $_GET['wcsal_product_ids'] ) : [];
        $selected_category_ids  = ! empty( $_GET['wcsal_category_ids'] ) ? array_map( 'intval', (array) $_GET['wcsal_category_ids'] ) : [];
        $selected_warehouse_ids = ! empty( $_GET['wcsal_warehouse_ids'] ) ? array_map( 'intval', (array) $_GET['wcsal_warehouse_ids'] ) : [];
        $selected_user_ids      = ! empty( $_GET['wcsal_user_ids'] ) ? array_map( 'intval', (array) $_GET['wcsal_user_ids'] ) : [];
        $selected_event_types   = ! empty( $_GET['wcsal_event_types'] ) ? (array) $_GET['wcsal_event_types'] : [];

        echo '<div id="wcsal-filters">';
        echo '<form method="get">';
        echo '<input type="hidden" name="page" value="wcsal_audit_log" />';

        // Product multi-select
        echo '<div class="wcsal-filter-group">';
        echo '<label for="wcsal_products">' . esc_html__( 'Product', 'wc-stock-audit-log' ) . '</label>';
        echo '<select id="wcsal_products" name="wcsal_product_ids[]" multiple="multiple" style="min-width:220px;">';
        // Populate selected product options.
        foreach ( $selected_product_ids as $pid ) {
            $product    = wc_get_product( $pid );
            $label_text = $product ? sprintf( '%s (#%d)', $product->get_name(), $pid ) : ( '#' . $pid );
            echo '<option value="' . esc_attr( $pid ) . '" selected>' . esc_html( $label_text ) . '</option>';
        }
        echo '</select>';
        echo '</div>';

        // Category multi-select
        echo '<div class="wcsal-filter-group">';
        echo '<label for="wcsal_categories">' . esc_html__( 'Category', 'wc-stock-audit-log' ) . '</label>';
        echo '<select id="wcsal_categories" name="wcsal_category_ids[]" multiple="multiple" style="min-width:220px;">';
        foreach ( $selected_category_ids as $cid ) {
            $cat        = get_term( $cid, 'product_cat' );
            $label_text = ( $cat && ! is_wp_error( $cat ) ) ? sprintf( '%s (#%d)', $cat->name, $cid ) : ( '#' . $cid );
            echo '<option value="' . esc_attr( $cid ) . '" selected>' . esc_html( $label_text ) . '</option>';
        }
        echo '</select>';
        echo '</div>';

        // User multi-select
        echo '<div class="wcsal-filter-group">';
        echo '<label for="wcsal_users">' . esc_html__( 'User', 'wc-stock-audit-log' ) . '</label>';
        echo '<select id="wcsal_users" name="wcsal_user_ids[]" multiple="multiple" style="min-width:220px;">';
        foreach ( $selected_user_ids as $uid ) {
            $user       = get_user_by( 'id', $uid );
            $label_text = $user ? sprintf( '%s (%s)', $user->display_name, $user->user_email ) : ( '#' . $uid );
            echo '<option value="' . esc_attr( $uid ) . '" selected>' . esc_html( $label_text ) . '</option>';
        }
        echo '</select>';
        echo '</div>';

        // Warehouse multi-select
        echo '<div class="wcsal-filter-group">';
        echo '<label for="wcsal_warehouses">' . esc_html__( 'Warehouse', 'wc-stock-audit-log' ) . '</label>';
        echo '<select id="wcsal_warehouses" name="wcsal_warehouse_ids[]" multiple="multiple" style="min-width:220px;">';
        foreach ( $selected_warehouse_ids as $wid ) {
            // Use filter to get friendly name; fall back to post title or ID.
            $name = apply_filters( 'wcsal_get_warehouse_name', '', $wid );
            if ( empty( $name ) ) {
                $post = get_post( $wid );
                $name = $post ? $post->post_title : '';
            }
            $label_text = $name ? sprintf( '%s (#%d)', $name, $wid ) : ( '#' . $wid );
            echo '<option value="' . esc_attr( $wid ) . '" selected>' . esc_html( $label_text ) . '</option>';
        }
        echo '</select>';
        echo '</div>';

        // Event type multi-select
        $all_event_types = [
            'sale'                 => __( 'Sale', 'wc-stock-audit-log' ),
            'refund'               => __( 'Refund', 'wc-stock-audit-log' ),
            'manual_adjustment'    => __( 'Manual Adjustment', 'wc-stock-audit-log' ),
            'openpos'              => __( 'OpenPOS', 'wc-stock-audit-log' ),
            'untracked_adjustment' => __( 'Untracked Adjustment', 'wc-stock-audit-log' ),
        ];
        echo '<div class="wcsal-filter-group">';
        echo '<label for="wcsal_event_types">' . esc_html__( 'Event Type', 'wc-stock-audit-log' ) . '</label>';
        echo '<select id="wcsal_event_types" name="wcsal_event_types[]" multiple="multiple" style="min-width:220px;">';
        foreach ( $all_event_types as $slug => $label ) {
            $selected = in_array( $slug, $selected_event_types, true ) ? ' selected' : '';
            echo '<option value="' . esc_attr( $slug ) . '"' . $selected . '>' . esc_html( $label ) . '</option>';
        }
        echo '</select>';
        echo '</div>';

        // Buttons
        echo '<div class="wcsal-filter-group">';
        submit_button( __( 'Apply Filters', 'wc-stock-audit-log' ), 'primary', '', false );
        echo '<a href="' . esc_url( admin_url( 'admin.php?page=wcsal_audit_log' ) ) . '" class="button" style="margin-left:8px;">' . esc_html__( 'Reset', 'wc-stock-audit-log' ) . '</a>';
        echo '</div>';

        echo '<div class="wcsal-filter-group">';
        submit_button( __( 'Export CSV (Server-Side)', 'wc-stock-audit-log' ), 'secondary', 'wcsal_export_csv', false );
        echo '</div>';

        echo '</form>';
        echo '</div>';
    }

    private function build_where_clause( &$params, $alias = 'l' ) {
        global $wpdb;

        $where   = 'WHERE 1=1';
        $params  = [];

        // Product IDs (multi-select)
        if ( ! empty( $_GET['wcsal_product_ids'] ) ) {
            $ids = array_map( 'intval', (array) $_GET['wcsal_product_ids'] );
            $ids = array_filter( $ids );
            if ( ! empty( $ids ) ) {
                $placeholders = implode( ',', array_fill( 0, count( $ids ), '%d' ) );
                $where       .= ' AND ' . $alias . '.product_id IN (' . $placeholders . ')';
                foreach ( $ids as $id ) {
                    $params[] = $id;
                }
            }
        }

        // Warehouse IDs (multi-select)
        if ( ! empty( $_GET['wcsal_warehouse_ids'] ) ) {
            $ids = array_map( 'intval', (array) $_GET['wcsal_warehouse_ids'] );
            $ids = array_filter( $ids );
            if ( ! empty( $ids ) ) {
                $placeholders = implode( ',', array_fill( 0, count( $ids ), '%d' ) );
                $where       .= ' AND ' . $alias . '.warehouse_id IN (' . $placeholders . ')';
                foreach ( $ids as $id ) {
                    $params[] = $id;
                }
            }
        }

        // User IDs (multi-select)
        if ( ! empty( $_GET['wcsal_user_ids'] ) ) {
            $ids = array_map( 'intval', (array) $_GET['wcsal_user_ids'] );
            $ids = array_filter( $ids );
            if ( ! empty( $ids ) ) {
                $placeholders = implode( ',', array_fill( 0, count( $ids ), '%d' ) );
                $where       .= ' AND ' . $alias . '.user_id IN (' . $placeholders . ')';
                foreach ( $ids as $id ) {
                    $params[] = $id;
                }
            }
        }

        // Event types (multi-select)
        if ( ! empty( $_GET['wcsal_event_types'] ) ) {
            $types = array_map( 'sanitize_key', (array) $_GET['wcsal_event_types'] );
            $types = array_filter( $types );
            if ( ! empty( $types ) ) {
                $placeholders = implode( ',', array_fill( 0, count( $types ), '%s' ) );
                $where       .= ' AND ' . $alias . '.event_type IN (' . $placeholders . ')';
                foreach ( $types as $type ) {
                    $params[] = $type;
                }
            }
        }

        // Category IDs (multi-select) with join against term tables
        if ( ! empty( $_GET['wcsal_category_ids'] ) ) {
            $ids = array_map( 'intval', (array) $_GET['wcsal_category_ids'] );
            $ids = array_filter( $ids );
            if ( ! empty( $ids ) ) {
                $placeholders = implode( ',', array_fill( 0, count( $ids ), '%d' ) );
                $where       .= ' AND tt.term_id IN (' . $placeholders . ') AND tt.taxonomy = %s';
                foreach ( $ids as $id ) {
                    $params[] = $id;
                }
                $params[] = 'product_cat';
            }
        }

        return $where;
    }

    private function get_join_clause() {
        global $wpdb;

        $join = '';
        // If categories filter is active (multi-select), join the term relationships and taxonomy tables
        if ( ! empty( $_GET['wcsal_category_ids'] ) ) {
            $tr  = $wpdb->term_relationships;
            $tt  = $wpdb->term_taxonomy;
            $log = $wpdb->prefix . 'wc_stock_audit_log';

            $join = " INNER JOIN {$tr} tr ON l.product_id = tr.object_id
                      INNER JOIN {$tt} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id";
        }

        return $join;
    }

    private function render_table() {
        global $wpdb;
        $table = $wpdb->prefix . 'wc_stock_audit_log';

        $params = [];
        $join   = $this->get_join_clause();
        $where  = $this->build_where_clause( $params, 'l' );

        $sql = "SELECT DISTINCT l.* FROM {$table} l {$join} {$where} ORDER BY l.created_at DESC, l.id DESC LIMIT 1000";

        $prepared = $wpdb->prepare( $sql, $params );
        $rows     = $wpdb->get_results( $prepared );

        echo '<table id="wcsal-log-table" class="widefat striped" style="margin-top:15px;">';
        echo '<thead><tr>';
        echo '<th>' . esc_html__( 'Date', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Product', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Variation ID', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Category', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Warehouse', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'User', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Change', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Before', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'After', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Event Type', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Source', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Source ID', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Note', 'wc-stock-audit-log' ) . '</th>';
        echo '</tr></thead>';
        echo '<tbody>';

        if ( $rows ) {
            foreach ( $rows as $row ) {
                $product_link = add_query_arg(
                    [
                        'page'       => 'wcsal_audit_log',
                        'wcsal_view' => 'product',
                        'product_id' => $row->product_id,
                    ],
                    admin_url( 'admin.php' )
                );
                $product_label = '#' . $row->product_id;
                $product_obj   = wc_get_product( $row->variation_id ?: $row->product_id );
                if ( $product_obj ) {
                    $product_label .= ' – ' . $product_obj->get_name();
                }

                $category_names = '';
                if ( $row->product_id ) {
                    $terms = get_the_terms( $row->product_id, 'product_cat' );
                    if ( ! is_wp_error( $terms ) && $terms ) {
                        $category_names = implode(
                            ', ',
                            wp_list_pluck( $terms, 'name' )
                        );
                    }
                }

                $user_display = '';
                if ( $row->user_id ) {
                    $user = get_user_by( 'id', $row->user_id );
                    if ( $user ) {
                        $user_display = $user->display_name;
                    }
                }

                echo '<tr>';
                echo '<td>' . esc_html( $row->created_at ) . '</td>';
                echo '<td><a href="' . esc_url( $product_link ) . '">' . esc_html( $product_label ) . '</a></td>';
                echo '<td>' . esc_html( $row->variation_id ) . '</td>';
                echo '<td>' . esc_html( $category_names ) . '</td>';
                // Show warehouse name if available; otherwise display the ID.
                $warehouse_name = '';
                if ( $row->warehouse_id ) {
                    // Allow third-party integrations to provide a friendly name for warehouses.
                    $warehouse_name = apply_filters( 'wcsal_get_warehouse_name', '', $row->warehouse_id );
                    if ( empty( $warehouse_name ) ) {
                        $wh_post = get_post( $row->warehouse_id );
                        if ( $wh_post ) {
                            $warehouse_name = $wh_post->post_title;
                        }
                    }
                }
                if ( empty( $warehouse_name ) ) {
                    $warehouse_display = '#' . (int) $row->warehouse_id;
                } else {
                    $warehouse_display = $warehouse_name . ' (#' . (int) $row->warehouse_id . ')';
                }
                echo '<td>' . esc_html( $warehouse_display ) . '</td>';
                echo '<td>' . esc_html( $user_display ) . '</td>';
                echo '<td>' . esc_html( $row->qty_change ) . '</td>';
                echo '<td>' . esc_html( null === $row->qty_before ? '-' : $row->qty_before ) . '</td>';
                echo '<td>' . esc_html( null === $row->qty_after ? '-' : $row->qty_after ) . '</td>';
                echo '<td>' . esc_html( $row->event_type ) . '</td>';
                echo '<td>' . esc_html( $row->event_source ) . '</td>';
                // For source identifiers associated with WooCommerce orders or refunds, link to the order edit screen.
                if ( ! empty( $row->source_id ) ) {
                    $source_id_val = $row->source_id;
                    $source_display = '';
                    // Determine if the event comes from an order or refund by inspecting the event source.
                    $is_order_related = false;
                    if ( strpos( $row->event_source, 'woocommerce_order' ) !== false || strpos( $row->event_source, 'woocommerce_refund' ) !== false || strpos( $row->event_source, 'woocommerce_credit' ) !== false ) {
                        $is_order_related = true;
                    }
                    if ( $is_order_related ) {
                        $order_link = admin_url( 'post.php?post=' . (int) $source_id_val . '&action=edit' );
                        $source_display = '<a href="' . esc_url( $order_link ) . '">#' . esc_html( $source_id_val ) . '</a>';
                    } else {
                        $source_display = esc_html( $source_id_val );
                    }
                    echo '<td>' . $source_display . '</td>';
                } else {
                    echo '<td>-</td>';
                }
                // Output note as HTML to allow links to orders; sanitize with wp_kses_post.
                echo '<td>' . wp_kses_post( $row->note ) . '</td>';
                echo '</tr>';
            }
        } else {
            // Do not use colspan as DataTables does not support colspan or rowspan.
            // Instead output the message in the first column and leave the remaining
            // cells empty so that each row has the same number of columns.
            echo '<tr>';
            echo '<td>' . esc_html__( 'No log entries found for this filter.', 'wc-stock-audit-log' ) . '</td>';
            // Output empty cells for the remaining 12 columns.
            for ( $i = 0; $i < 12; $i++ ) {
                echo '<td>&nbsp;</td>';
            }
            echo '</tr>';
        }

        echo '</tbody>';
        echo '</table>';
    }

    /**
     * Product-level detail timeline with running total (from 0).
     */
    private function render_product_timeline( $product_id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'wc_stock_audit_log';

        $product_id = (int) $product_id;
        $product    = wc_get_product( $product_id );

        echo '<h2 style="margin-top:20px;">' . esc_html__( 'Product Timeline', 'wc-stock-audit-log' ) . '</h2>';

        if ( $product ) {
            echo '<p><strong>' . esc_html( $product->get_name() ) . '</strong> (ID: ' . esc_html( $product_id ) . ')</p>';
        } else {
            echo '<p>' . sprintf( esc_html__( 'Product ID: %d', 'wc-stock-audit-log' ), $product_id ) . '</p>';
        }

        $back_link = add_query_arg(
            [
                'page' => 'wcsal_audit_log',
            ],
            admin_url( 'admin.php' )
        );

        echo '<p><a href="' . esc_url( $back_link ) . '" class="button">&larr; ' . esc_html__( 'Back to Audit Log', 'wc-stock-audit-log' ) . '</a></p>';

        $sql = $wpdb->prepare(
            "SELECT * FROM {$table}
             WHERE product_id = %d
             ORDER BY created_at ASC, id ASC",
            $product_id
        );

        $rows = $wpdb->get_results( $sql );

        echo '<table id="wcsal-product-timeline" class="widefat striped" style="margin-top:15px;">';
        echo '<thead><tr>';
        echo '<th>' . esc_html__( 'Date', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Warehouse', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'User', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Change', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Running Total (from 0)', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Event Type', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Source', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Source ID', 'wc-stock-audit-log' ) . '</th>';
        echo '<th>' . esc_html__( 'Note', 'wc-stock-audit-log' ) . '</th>';
        echo '</tr></thead>';
        echo '<tbody>';

        if ( $rows ) {
            $running_total = 0.0;

            foreach ( $rows as $row ) {
                $running_total += (float) $row->qty_change;

                $user_display = '';
                if ( $row->user_id ) {
                    $user = get_user_by( 'id', $row->user_id );
                    if ( $user ) {
                        $user_display = $user->display_name;
                    }
                }

                echo '<tr>';
                echo '<td>' . esc_html( $row->created_at ) . '</td>';
                // Render warehouse name if available; otherwise use the ID.
                $wh_name = '';
                if ( $row->warehouse_id ) {
                    $wh_name = apply_filters( 'wcsal_get_warehouse_name', '', $row->warehouse_id );
                    if ( empty( $wh_name ) ) {
                        $wh_post = get_post( $row->warehouse_id );
                        if ( $wh_post ) {
                            $wh_name = $wh_post->post_title;
                        }
                    }
                }
                $wh_display = empty( $wh_name ) ? '#' . (int) $row->warehouse_id : $wh_name . ' (#' . (int) $row->warehouse_id . ')';
                echo '<td>' . esc_html( $wh_display ) . '</td>';
                echo '<td>' . esc_html( $user_display ) . '</td>';
                echo '<td>' . esc_html( $row->qty_change ) . '</td>';
                echo '<td>' . esc_html( $running_total ) . '</td>';
                echo '<td>' . esc_html( $row->event_type ) . '</td>';
                echo '<td>' . esc_html( $row->event_source ) . '</td>';
                // Source ID column – link to order edit page if event relates to an order/refund.
                if ( ! empty( $row->source_id ) ) {
                    $src_id = $row->source_id;
                    $src_display = '';
                    $is_order_related = false;
                    if ( strpos( $row->event_source, 'woocommerce_order' ) !== false || strpos( $row->event_source, 'woocommerce_refund' ) !== false ) {
                        $is_order_related = true;
                    }
                    if ( $is_order_related ) {
                        $order_link = admin_url( 'post.php?post=' . (int) $src_id . '&action=edit' );
                        $src_display = '<a href="' . esc_url( $order_link ) . '">#' . esc_html( $src_id ) . '</a>';
                    } else {
                        $src_display = esc_html( $src_id );
                    }
                    echo '<td>' . $src_display . '</td>';
                } else {
                    echo '<td>-</td>';
                }
                echo '<td>' . esc_html( $row->note ) . '</td>';
                echo '</tr>';
            }
        } else {
            // DataTables does not support colspan, so fill each column with empty
            // values and show the message in the first column.
            echo '<tr>';
            echo '<td>' . esc_html__( 'No log entries found for this product.', 'wc-stock-audit-log' ) . '</td>';
            // There are 8 additional columns in this table.
            for ( $i = 0; $i < 8; $i++ ) {
                echo '<td>&nbsp;</td>';
            }
            echo '</tr>';
        }

        echo '</tbody>';
        echo '</table>';
    }

    private function render_rebuild_box() {
        $job = WCSAL_Rebuilder::get_current_job();

        echo '<hr />';
        echo '<h2 style="margin-top:25px;">' . esc_html__( 'Retrospective Rebuild (Pro)', 'wc-stock-audit-log' ) . '</h2>';
        echo '<p>' . esc_html__( 'Rebuild stock movements from historical WooCommerce orders and refunds using a safe, batched process. Leave dates empty to process all history.', 'wc-stock-audit-log' ) . '</p>';

        if ( $job ) {
            echo '<div id="wcsal-rebuild-status" class="notice" style="padding:10px;margin-bottom:10px;">';

            if ( 'running' === $job['status'] ) {
                echo '<p><strong>' . esc_html__( 'Job status:', 'wc-stock-audit-log' ) . '</strong> ' . esc_html__( 'Running', 'wc-stock-audit-log' ) . '</p>';
            } elseif ( 'finished' === $job['status'] ) {
                echo '<p><strong>' . esc_html__( 'Job status:', 'wc-stock-audit-log' ) . '</strong> ' . esc_html__( 'Finished', 'wc-stock-audit-log' ) . '</p>';
                echo '<p>' . esc_html__( 'You can run a new rebuild at any time; previous rebuilt entries will be cleared.', 'wc-stock-audit-log' ) . '</p>';
            }

            $processed = (int) $job['processed'];
            $total     = (int) $job['total_orders'];
            $percent   = ( $total > 0 ) ? min( 100, round( ( $processed / $total ) * 100 ) ) : 0;

            echo '<p>' . sprintf( esc_html__( 'Processed %1$d of %2$d orders (%3$d%%).', 'wc-stock-audit-log' ), $processed, $total, $percent ) . '</p>';
            if ( ! empty( $job['last_message'] ) ) {
                echo '<p>' . esc_html( $job['last_message'] ) . '</p>';
            }

            echo '<div style="background:#eee;border:1px solid #ccc;width:300px;height:16px;position:relative;">';
            echo '<div id="wcsal-rebuild-progress-bar" style="background:#46b450;height:100%;width:' . esc_attr( $percent ) . '%;"></div>';
            echo '</div>';

            echo '</div>';
        }

        echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" style="margin-top:10px;">';
        wp_nonce_field( 'wcsal_rebuild_history' );
        echo '<input type="hidden" name="action" value="wcsal_rebuild_history" />';

        echo '<table class="form-table"><tbody>';
        echo '<tr>';
        echo '<th scope="row"><label for="wcsal_from_date">' . esc_html__( 'From Date (YYYY-MM-DD)', 'wc-stock-audit-log' ) . '</label></th>';
        echo '<td><input type="text" name="wcsal_from_date" id="wcsal_from_date" placeholder="2020-01-01" /></td>';
        echo '</tr>';

        echo '<tr>';
        echo '<th scope="row"><label for="wcsal_to_date">' . esc_html__( 'To Date (YYYY-MM-DD)', 'wc-stock-audit-log' ) . '</label></th>';
        echo '<td><input type="text" name="wcsal_to_date" id="wcsal_to_date" placeholder="2025-12-31" /></td>';
        echo '</tr>';
        echo '</tbody></table>';

        submit_button( __( 'Start Batched Rebuild', 'wc-stock-audit-log' ), 'primary', '', false );

        echo '</form>';
    }

    private function render_pro_upsell() {
        echo '<hr />';
        echo '<h2 style="margin-top:25px;">' . esc_html__( 'Retrospective Rebuild (Pro Feature)', 'wc-stock-audit-log' ) . '</h2>';
        echo '<p>';
        echo esc_html__( 'Upgrade to the Pro version to unlock:', 'wc-stock-audit-log' );
        echo '</p><ul style="list-style:disc;margin-left:20px;">';
        echo '<li>' . esc_html__( 'Full retroactive stock history rebuilding from all past orders & refunds (batched, safe for large stores).', 'wc-stock-audit-log' ) . '</li>';
        echo '<li>' . esc_html__( 'Multi-warehouse, OpenPOS-aware movements reconstruction.', 'wc-stock-audit-log' ) . '</li>';
        echo '<li>' . esc_html__( 'Automatic mismatch detection and email alerts.', 'wc-stock-audit-log' ) . '</li>';
        echo '<li>' . esc_html__( 'Per-product stock movement timelines & advanced exports.', 'wc-stock-audit-log' ) . '</li>';
        echo '</ul>';
        // Link to the official product page.  Use sparkcutlabs.com rather than a
        // placeholder domain to ensure users are sent to the correct site.  This URL
        // has been updated to point to the product’s info page per request.
        echo '<p><a href="https://sparkcutlabs.com/product-stock-audit.html" target="_blank" class="button button-primary">' . esc_html__( 'View Premium Features', 'wc-stock-audit-log' ) . '</a></p>';

        echo '<p><em>' . esc_html__( 'Pro activation is controlled by your central license manager. This plugin honours your suite licensing rules (via filters / helper functions). If you have received a licence key for this plugin you can enter it below to unlock Pro functionality.', 'wc-stock-audit-log' ) . '</em></p>';

        // Licence entry form.  Uses the same settings group as other options to
        // persist the licence key.  If a key is present then the plugin will
        // behave as though the Pro version is activated via WCSAL_Licensing.
        echo '<form method="post" action="options.php" style="margin-top:10px;">';
        // Output nonce, option_group and settings_fields.  The group must
        // match that used in register_settings().
        settings_fields( 'wcsal_settings' );
        $license = get_option( WCSAL_LICENSE_OPTION, '' );
        echo '<table class="form-table"><tbody>';
        echo '<tr>';
        echo '<th scope="row">' . esc_html__( 'Licence Key', 'wc-stock-audit-log' ) . '</th>';
        echo '<td><input type="text" name="' . esc_attr( WCSAL_LICENSE_OPTION ) . '" value="' . esc_attr( $license ) . '" class="regular-text" placeholder="Enter your licence key" /></td>';
        echo '</tr>';
        echo '</tbody></table>';
        submit_button( __( 'Save Licence', 'wc-stock-audit-log' ), 'primary' );
        echo '</form>';
    }

    private function render_settings_box() {
        echo '<hr />';
        echo '<h2 style="margin-top:25px;">' . esc_html__( 'Audit Settings', 'wc-stock-audit-log' ) . '</h2>';

        echo '<form method="post" action="options.php">';
        settings_fields( 'wcsal_settings' );

        $enable_email = (bool) get_option( 'wcsal_enable_mismatch_email', false );
        $recipient    = get_option( 'wcsal_mismatch_email_recipient', get_option( 'admin_email' ) );

        echo '<table class="form-table">';
        echo '<tr>';
        echo '<th scope="row">' . esc_html__( 'Email alerts for mismatches', 'wc-stock-audit-log' ) . '</th>';
        echo '<td>';
        echo '<label><input type="checkbox" name="wcsal_enable_mismatch_email" value="1" ' . checked( $enable_email, true, false ) . ' /> ' . esc_html__( 'Send an email when a mismatch vs reconstructed history is detected.', 'wc-stock-audit-log' ) . '</label>';
        echo '</td>';
        echo '</tr>';

        echo '<tr>';
        echo '<th scope="row">' . esc_html__( 'Alert recipient', 'wc-stock-audit-log' ) . '</th>';
        echo '<td>';
        echo '<input type="email" name="wcsal_mismatch_email_recipient" value="' . esc_attr( $recipient ) . '" class="regular-text" />';
        echo '</td>';
        echo '</tr>';
        echo '</table>';

        submit_button();

        echo '</form>';
    }

    // -----------------------------------------------------
    // CSV Export (server-side; in addition to DataTables CSV/Excel)
    // -----------------------------------------------------
    private function maybe_export_csv() {
        if ( empty( $_GET['wcsal_export_csv'] ) ) {
            return;
        }

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'wc_stock_audit_log';

        $params = [];
        $join   = $this->get_join_clause();
        $where  = $this->build_where_clause( $params, 'l' );

        $sql      = "SELECT DISTINCT l.* FROM {$table} l {$join} {$where} ORDER BY l.created_at DESC, l.id DESC LIMIT 5000";
        $prepared = $wpdb->prepare( $sql, $params );
        $rows     = $wpdb->get_results( $prepared, ARRAY_A );

        nocache_headers();
        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=stock-audit-log-' . date( 'Ymd-His' ) . '.csv' );

        $output = fopen( 'php://output', 'w' );
        fputcsv(
            $output,
            [
                'id',
                'created_at',
                'product_id',
                'variation_id',
                'warehouse_id',
                'user_id',
                'qty_change',
                'qty_before',
                'qty_after',
                'event_type',
                'event_source',
                'source_id',
                'note',
            ]
        );

        if ( $rows ) {
            foreach ( $rows as $row ) {
                fputcsv(
                    $output,
                    [
                        $row['id'],
                        $row['created_at'],
                        $row['product_id'],
                        $row['variation_id'],
                        $row['warehouse_id'],
                        $row['user_id'],
                        $row['qty_change'],
                        $row['qty_before'],
                        $row['qty_after'],
                        $row['event_type'],
                        $row['event_source'],
                        $row['source_id'],
                        $row['note'],
                    ]
                );
            }
        }

        fclose( $output );
        exit;
    }

    // -----------------------------------------------------
    // Inline JS: DataTables + Autocomplete + Rebuild progress
    // -----------------------------------------------------
    private function render_js() {
        if ( 'woocommerce_page_wcsal_audit_log' !== get_current_screen()->id ) {
            return;
        }

        $search_nonce = wp_create_nonce( 'wcsal_search' );
        $rebuild_nonce = wp_create_nonce( 'wcsal_rebuild_step' );
        $job = class_exists( 'WCSAL_Rebuilder' ) ? WCSAL_Rebuilder::get_current_job() : null;
        $job_running = $job && isset( $job['status'] ) && 'running' === $job['status'];

        ?>
        <script type="text/javascript">
        (function($){
            $(document).ready(function(){

                // DataTables for main log table
                if ($('#wcsal-log-table').length && $.fn.DataTable) {
                    $('#wcsal-log-table').DataTable({
                        pageLength: 25,
                        order: [[0, 'desc']],
                        dom: 'Bfrtip',
                        buttons: [
                            {
                                extend: 'csvHtml5',
                                title: 'stock-audit-log'
                            },
                            {
                                extend: 'excelHtml5',
                                title: 'stock-audit-log'
                            }
                        ]
                    });
                }

                // DataTables for product timeline
                if ($('#wcsal-product-timeline').length && $.fn.DataTable) {
                    $('#wcsal-product-timeline').DataTable({
                        pageLength: 50,
                        order: [[0, 'asc']],
                        dom: 'Bfrtip',
                        buttons: [
                            {
                                extend: 'csvHtml5',
                                title: 'stock-audit-product-timeline'
                            },
                            {
                                extend: 'excelHtml5',
                                title: 'stock-audit-product-timeline'
                            }
                        ]
                    });
                }

                // Autocomplete: helper factory
                function makeAutocomplete(inputSelector, hiddenSelector, action) {
                    $(inputSelector).autocomplete({
                        minLength: 2,
                        source: function(request, response){
                            $.getJSON(ajaxurl, {
                                action: action,
                                term: request.term,
                                nonce: '<?php echo esc_js( $search_nonce ); ?>'
                            }, function(data){
                                response(data || []);
                            });
                        },
                        select: function(event, ui){
                            $(hiddenSelector).val(ui.item.id);
                        },
                        change: function(event, ui){
                            if (!ui.item) {
                                $(hiddenSelector).val('');
                            }
                        }
                    });
                }

                // Initialise Select2 controls for multi-select filters.
                if ($.fn.select2) {
                    // Products
                    $('#wcsal_products').select2({
                        ajax: {
                            url: ajaxurl,
                            dataType: 'json',
                            delay: 250,
                            data: function(params) {
                                return {
                                    action: 'wcsal_search_products',
                                    term: params.term,
                                    nonce: '<?php echo esc_js( $search_nonce ); ?>'
                                };
                            },
                            processResults: function(data) {
                                return {
                                    results: $.map(data, function(item) {
                                        return { id: item.id, text: item.label };
                                    })
                                };
                            },
                            cache: true
                        },
                        width: 'resolve',
                        placeholder: '<?php echo esc_js( __( 'Search product…', 'wc-stock-audit-log' ) ); ?>',
                        allowClear: true,
                        multiple: true
                    });
                    // Categories
                    $('#wcsal_categories').select2({
                        ajax: {
                            url: ajaxurl,
                            dataType: 'json',
                            delay: 250,
                            data: function(params) {
                                return {
                                    action: 'wcsal_search_categories',
                                    term: params.term,
                                    nonce: '<?php echo esc_js( $search_nonce ); ?>'
                                };
                            },
                            processResults: function(data) {
                                return {
                                    results: $.map(data, function(item) {
                                        return { id: item.id, text: item.label };
                                    })
                                };
                            },
                            cache: true
                        },
                        width: 'resolve',
                        placeholder: '<?php echo esc_js( __( 'Search category…', 'wc-stock-audit-log' ) ); ?>',
                        allowClear: true,
                        multiple: true
                    });
                    // Users
                    $('#wcsal_users').select2({
                        ajax: {
                            url: ajaxurl,
                            dataType: 'json',
                            delay: 250,
                            data: function(params) {
                                return {
                                    action: 'wcsal_search_users',
                                    term: params.term,
                                    nonce: '<?php echo esc_js( $search_nonce ); ?>'
                                };
                            },
                            processResults: function(data) {
                                return {
                                    results: $.map(data, function(item) {
                                        return { id: item.id, text: item.label };
                                    })
                                };
                            },
                            cache: true
                        },
                        width: 'resolve',
                        placeholder: '<?php echo esc_js( __( 'Search user…', 'wc-stock-audit-log' ) ); ?>',
                        allowClear: true,
                        multiple: true
                    });
                    // Warehouses
                    $('#wcsal_warehouses').select2({
                        ajax: {
                            url: ajaxurl,
                            dataType: 'json',
                            delay: 250,
                            data: function(params) {
                                return {
                                    action: 'wcsal_search_warehouses',
                                    term: params.term,
                                    nonce: '<?php echo esc_js( $search_nonce ); ?>'
                                };
                            },
                            processResults: function(data) {
                                return {
                                    results: $.map(data, function(item) {
                                        return { id: item.id, text: item.label };
                                    })
                                };
                            },
                            cache: true
                        },
                        width: 'resolve',
                        placeholder: '<?php echo esc_js( __( 'Search warehouse…', 'wc-stock-audit-log' ) ); ?>',
                        allowClear: true,
                        multiple: true
                    });
                    // Event types (static options, no AJAX)
                    $('#wcsal_event_types').select2({
                        width: 'resolve',
                        placeholder: '<?php echo esc_js( __( 'Select event type(s)', 'wc-stock-audit-log' ) ); ?>',
                        allowClear: true,
                        multiple: true
                    });
                }

                // Rebuild progress polling
                var jobRunning = <?php echo $job_running ? 'true' : 'false'; ?>;
                if (jobRunning) {
                    function wcsalRunRebuildStep() {
                        $.post(ajaxurl, {
                            action: 'wcsal_rebuild_step',
                            nonce: '<?php echo esc_js( $rebuild_nonce ); ?>'
                        }, function(resp){
                            if (!resp || !resp.success || !resp.data) {
                                return;
                            }
                            var data = resp.data;

                            if ($('#wcsal-rebuild-status').length && typeof data.percent !== 'undefined') {
                                var percent = data.percent || 0;
                                $('#wcsal-rebuild-progress-bar').css('width', percent + '%');
                                // Update message line (2nd <p>)
                                $('#wcsal-rebuild-status').find('p').eq(1).text(
                                    data.message || ''
                                );
                            }

                            if (!data.done) {
                                setTimeout(wcsalRunRebuildStep, 500);
                            } else {
                                window.location = window.location.href;
                            }
                        });
                    }
                    wcsalRunRebuildStep();
                }
            });
        })(jQuery);
        </script>
        <?php
    }
}
