<?php
/**
 * Plugin Name: WooCommerce Shipping Cutoff Countdown
 * Plugin URI: https://your-site.com
 * Description: Show a countdown to today\'s shipping cutoff time on product, cart, and checkout pages. Pro adds advanced rules, holidays, per-country cutoffs and per-product overrides.
 * Version: 1.0.0
 * Author: Sparkcut Labs
 * Author URI: https://sparkcutlabs.com
 * Text Domain: wcscc
 * Requires PHP: 7.4
 * Requires at least: 5.8
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'WCSCC_VERSION', '1.0.0' );
define( 'WCSCC_FILE', __FILE__ );
define( 'WCSCC_DIR', plugin_dir_path( __FILE__ ) );
define( 'WCSCC_URL', plugin_dir_url( __FILE__ ) );

// License server endpoint (Sparkcut Labs license server api.php).
// Example: https://licensing.yourdomain.com/api.php
define( 'WCSCC_LICENSE_API', 'http://sparkcutlabs.com/license-server/api.php' );

// Product slug as used in your license server (e.g. shipping_cutoff_pro).
define( 'WCSCC_PRODUCT_SLUG', 'shipping_cutoff_pro' );

/**
 * Shipping Cutoff License client (Sparkcut-compatible)
 */
class WCSCC_License {

    const OPT_KEY         = 'wcscc_license';
    const TRANSIENT_STATE = 'wcscc_license_status';
    const CHECK_INTERVAL  = 12 * HOUR_IN_SECONDS;
    const GRACE_HOURS     = 72;

    public function init() {
        if ( is_admin() ) {
            add_action( 'admin_menu', [ $this, 'register_menu' ] );
            add_action( 'admin_init', [ $this, 'maybe_handle_form' ] );
        }
    }

    public static function is_pro() {
        $status = self::get_cached_status();
        return in_array( $status, [ 'valid', 'grace' ], true );
    }

    public static function get_cached_status() {
        $status = get_transient( self::TRANSIENT_STATE );
        if ( $status ) {
            return $status;
        }

        $opt = get_option(
            self::OPT_KEY,
            [
                'key'         => '',
                'activated'   => false,
                'grace_until' => 0,
                'last_check'  => 0,
            ]
        );

        if ( empty( $opt['key'] ) ) {
            set_transient( self::TRANSIENT_STATE, 'invalid', HOUR_IN_SECONDS );
            return 'invalid';
        }

        $now = time();
        if ( ! empty( $opt['last_check'] ) && ( $now - (int) $opt['last_check'] ) < self::CHECK_INTERVAL ) {
            if ( ! empty( $opt['activated'] ) ) {
                set_transient( self::TRANSIENT_STATE, 'valid', HOUR_IN_SECONDS );
                return 'valid';
            }
            if ( ! empty( $opt['grace_until'] ) && $opt['grace_until'] > $now ) {
                set_transient( self::TRANSIENT_STATE, 'grace', HOUR_IN_SECONDS );
                return 'grace';
            }
            set_transient( self::TRANSIENT_STATE, 'invalid', HOUR_IN_SECONDS );
            return 'invalid';
        }

        // Silent background check.
        $self = new self();
        $res  = $self->remote_request( $opt['key'], 'validate' );
        $self->handle_validation_result( $res, $opt, false );

        $status = get_transient( self::TRANSIENT_STATE );
        return $status ? $status : 'unknown';
    }

    public function register_menu() {
        add_submenu_page(
            'woocommerce',
            __( 'Shipping Cutoff Countdown', 'wcscc' ),
            __( 'Shipping Cutoff', 'wcscc' ),
            'manage_woocommerce',
            'wcscc-settings',
            [ $this, 'render_settings_page' ]
        );
    }

    public function maybe_handle_form() {
        if ( empty( $_POST['wcscc_settings_nonce'] ) ) {
            return;
        }

        if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['wcscc_settings_nonce'] ) ), 'wcscc_save_settings' ) ) {
            return;
        }

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        // General settings.
        $settings = [
            'base_cutoff_time'     => isset( $_POST['wcscc_base_cutoff_time'] ) ? sanitize_text_field( wp_unslash( $_POST['wcscc_base_cutoff_time'] ) ) : '16:00',
            'enable_on_product'    => ! empty( $_POST['wcscc_enable_on_product'] ) ? 1 : 0,
            'enable_on_cart'       => ! empty( $_POST['wcscc_enable_on_cart'] ) ? 1 : 0,
            'enable_on_checkout'   => ! empty( $_POST['wcscc_enable_on_checkout'] ) ? 1 : 0,
            'message_template'     => isset( $_POST['wcscc_message_template'] ) ? wp_kses_post( wp_unslash( $_POST['wcscc_message_template'] ) ) : __( 'Order within {countdown} for dispatch today.', 'wcscc' ),
            'fallback_template'    => isset( $_POST['wcscc_fallback_template'] ) ? wp_kses_post( wp_unslash( $_POST['wcscc_fallback_template'] ) ) : __( 'Order now for next business day dispatch.', 'wcscc' ),

            // Pro options.
            'enable_weekday_rules' => ! empty( $_POST['wcscc_enable_weekday_rules'] ) ? 1 : 0,
            'cutoff_monday'        => isset( $_POST['wcscc_cutoff_monday'] ) ? sanitize_text_field( wp_unslash( $_POST['wcscc_cutoff_monday'] ) ) : '',
            'cutoff_tuesday'       => isset( $_POST['wcscc_cutoff_tuesday'] ) ? sanitize_text_field( wp_unslash( $_POST['wcscc_cutoff_tuesday'] ) ) : '',
            'cutoff_wednesday'     => isset( $_POST['wcscc_cutoff_wednesday'] ) ? sanitize_text_field( wp_unslash( $_POST['wcscc_cutoff_wednesday'] ) ) : '',
            'cutoff_thursday'      => isset( $_POST['wcscc_cutoff_thursday'] ) ? sanitize_text_field( wp_unslash( $_POST['wcscc_cutoff_thursday'] ) ) : '',
            'cutoff_friday'        => isset( $_POST['wcscc_cutoff_friday'] ) ? sanitize_text_field( wp_unslash( $_POST['wcscc_cutoff_friday'] ) ) : '',
            'cutoff_saturday'      => isset( $_POST['wcscc_cutoff_saturday'] ) ? sanitize_text_field( wp_unslash( $_POST['wcscc_cutoff_saturday'] ) ) : '',
            'cutoff_sunday'        => isset( $_POST['wcscc_cutoff_sunday'] ) ? sanitize_text_field( wp_unslash( $_POST['wcscc_cutoff_sunday'] ) ) : '',
            'holiday_dates'        => isset( $_POST['wcscc_holiday_dates'] ) ? sanitize_textarea_field( wp_unslash( $_POST['wcscc_holiday_dates'] ) ) : '',
            'enable_badges'        => ! empty( $_POST['wcscc_enable_badges'] ) ? 1 : 0,
            'badge_today'          => isset( $_POST['wcscc_badge_today'] ) ? sanitize_text_field( wp_unslash( $_POST['wcscc_badge_today'] ) ) : __( 'Ships today', 'wcscc' ),
            'badge_tomorrow'       => isset( $_POST['wcscc_badge_tomorrow'] ) ? sanitize_text_field( wp_unslash( $_POST['wcscc_badge_tomorrow'] ) ) : __( 'Ships tomorrow', 'wcscc' ),
            'badge_later'          => isset( $_POST['wcscc_badge_later'] ) ? sanitize_text_field( wp_unslash( $_POST['wcscc_badge_later'] ) ) : __( 'Ships on {date}', 'wcscc' ),
            'enable_country_rules' => ! empty( $_POST['wcscc_enable_country_rules'] ) ? 1 : 0,
            'country_rules'        => isset( $_POST['wcscc_country_rules'] ) ? sanitize_textarea_field( wp_unslash( $_POST['wcscc_country_rules'] ) ) : '',
        ];

        update_option( 'wcscc_settings', $settings );

        // License key.
        if ( isset( $_POST['wcscc_license_key'] ) ) {
            $key = trim( sanitize_text_field( wp_unslash( $_POST['wcscc_license_key'] ) ) );
            $opt = get_option(
                self::OPT_KEY,
                [
                    'key'         => '',
                    'activated'   => false,
                    'grace_until' => 0,
                    'last_check'  => 0,
                ]
            );

            if ( $key !== $opt['key'] ) {
                $res = $this->remote_request( $key, 'activate' );
                $this->handle_validation_result( $res, $opt, true, $key );
            } else {
                $res = $this->remote_request( $key, 'validate' );
                $this->handle_validation_result( $res, $opt, true );
            }
        }

        wp_safe_redirect(
            add_query_arg(
                [
                    'page'    => 'wcscc-settings',
                    'updated' => 'true',
                ],
                admin_url( 'admin.php' )
            )
        );
        exit;
    }

    public function render_settings_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        $defaults = [
            'base_cutoff_time'     => '16:00',
            'enable_on_product'    => 1,
            'enable_on_cart'       => 0,
            'enable_on_checkout'   => 0,
            'message_template'     => __( 'Order within {countdown} for dispatch today.', 'wcscc' ),
            'fallback_template'    => __( 'Order now for next business day dispatch.', 'wcscc' ),
            'enable_weekday_rules' => 0,
            'cutoff_monday'        => '',
            'cutoff_tuesday'       => '',
            'cutoff_wednesday'     => '',
            'cutoff_thursday'      => '',
            'cutoff_friday'        => '',
            'cutoff_saturday'      => '',
            'cutoff_sunday'        => '',
            'holiday_dates'        => '',
            'enable_badges'        => 0,
            'badge_today'          => __( 'Ships today', 'wcscc' ),
            'badge_tomorrow'       => __( 'Ships tomorrow', 'wcscc' ),
            'badge_later'          => __( 'Ships on {date}', 'wcscc' ),
            'enable_country_rules' => 0,
            'country_rules'        => '',
        ];

        $settings = get_option( 'wcscc_settings', [] );
        $settings = wp_parse_args( $settings, $defaults );

        $opt    = get_option(
            self::OPT_KEY,
            [
                'key'         => '',
                'activated'   => false,
                'grace_until' => 0,
                'last_check'  => 0,
            ]
        );
        $status = self::get_cached_status();
        $domain = $this->domain();
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Shipping Cutoff Countdown', 'wcscc' ); ?></h1>

            <?php if ( isset( $_GET['updated'] ) ) : ?>
                <div class="notice notice-success"><p><?php esc_html_e( 'Settings saved.', 'wcscc' ); ?></p></div>
            <?php endif; ?>

            <form method="post">
                <?php wp_nonce_field( 'wcscc_save_settings', 'wcscc_settings_nonce' ); ?>

                <h2><?php esc_html_e( 'General', 'wcscc' ); ?></h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="wcscc_base_cutoff_time"><?php esc_html_e( 'Daily cutoff time', 'wcscc' ); ?></label></th>
                        <td>
                            <input type="text" id="wcscc_base_cutoff_time" name="wcscc_base_cutoff_time" value="<?php echo esc_attr( $settings['base_cutoff_time'] ); ?>" />
                            <p class="description"><?php esc_html_e( 'Format: HH:MM in store timezone (e.g. 16:00).', 'wcscc' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Display locations', 'wcscc' ); ?></th>
                        <td>
                            <label><input type="checkbox" name="wcscc_enable_on_product" value="1" <?php checked( $settings['enable_on_product'], 1 ); ?> /> <?php esc_html_e( 'Single product pages', 'wcscc' ); ?></label><br/>
                            <label><input type="checkbox" name="wcscc_enable_on_cart" value="1" <?php checked( $settings['enable_on_cart'], 1 ); ?> /> <?php esc_html_e( 'Cart page', 'wcscc' ); ?></label><br/>
                            <label><input type="checkbox" name="wcscc_enable_on_checkout" value="1" <?php checked( $settings['enable_on_checkout'], 1 ); ?> /> <?php esc_html_e( 'Checkout page', 'wcscc' ); ?></label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wcscc_message_template"><?php esc_html_e( 'Countdown message', 'wcscc' ); ?></label></th>
                        <td>
                            <textarea id="wcscc_message_template" name="wcscc_message_template" rows="3" cols="60"><?php echo esc_textarea( $settings['message_template'] ); ?></textarea>
                            <p class="description"><?php esc_html_e( 'Use {countdown} for remaining time, {time} for cutoff time, {date} for date.', 'wcscc' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wcscc_fallback_template"><?php esc_html_e( 'Fallback message (after cutoff)', 'wcscc' ); ?></label></th>
                        <td>
                            <textarea id="wcscc_fallback_template" name="wcscc_fallback_template" rows="3" cols="60"><?php echo esc_textarea( $settings['fallback_template'] ); ?></textarea>
                            <p class="description"><?php esc_html_e( 'Shown when cutoff has passed or on holidays/weekends.', 'wcscc' ); ?></p>
                        </td>
                    </tr>
                </table>

                <hr/>

                <h2><?php esc_html_e( 'Pro – Advanced Schedules', 'wcscc' ); ?></h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Weekday-specific cutoffs', 'wcscc' ); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wcscc_enable_weekday_rules" value="1" <?php checked( $settings['enable_weekday_rules'], 1 ); ?> />
                                <?php esc_html_e( 'Use different cutoff times per weekday.', 'wcscc' ); ?>
                            </label>
                            <?php if ( ! self::is_pro() ) : ?>
                                <p class="description"><em><?php esc_html_e( 'Requires an active Pro license.', 'wcscc' ); ?></em></p>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Weekday cutoff times', 'wcscc' ); ?></th>
                        <td>
                            <table>
                                <tr>
                                    <td><?php esc_html_e( 'Monday', 'wcscc' ); ?></td>
                                    <td><input type="text" name="wcscc_cutoff_monday" value="<?php echo esc_attr( $settings['cutoff_monday'] ); ?>" placeholder="16:00" /></td>
                                </tr>
                                <tr>
                                    <td><?php esc_html_e( 'Tuesday', 'wcscc' ); ?></td>
                                    <td><input type="text" name="wcscc_cutoff_tuesday" value="<?php echo esc_attr( $settings['cutoff_tuesday'] ); ?>" placeholder="16:00" /></td>
                                </tr>
                                <tr>
                                    <td><?php esc_html_e( 'Wednesday', 'wcscc' ); ?></td>
                                    <td><input type="text" name="wcscc_cutoff_wednesday" value="<?php echo esc_attr( $settings['cutoff_wednesday'] ); ?>" placeholder="16:00" /></td>
                                </tr>
                                <tr>
                                    <td><?php esc_html_e( 'Thursday', 'wcscc' ); ?></td>
                                    <td><input type="text" name="wcscc_cutoff_thursday" value="<?php echo esc_attr( $settings['cutoff_thursday'] ); ?>" placeholder="16:00" /></td>
                                </tr>
                                <tr>
                                    <td><?php esc_html_e( 'Friday', 'wcscc' ); ?></td>
                                    <td><input type="text" name="wcscc_cutoff_friday" value="<?php echo esc_attr( $settings['cutoff_friday'] ); ?>" placeholder="14:00" /></td>
                                </tr>
                                <tr>
                                    <td><?php esc_html_e( 'Saturday', 'wcscc' ); ?></td>
                                    <td><input type="text" name="wcscc_cutoff_saturday" value="<?php echo esc_attr( $settings['cutoff_saturday'] ); ?>" placeholder="12:00" /></td>
                                </tr>
                                <tr>
                                    <td><?php esc_html_e( 'Sunday', 'wcscc' ); ?></td>
                                    <td><input type="text" name="wcscc_cutoff_sunday" value="<?php echo esc_attr( $settings['cutoff_sunday'] ); ?>" placeholder="closed" /></td>
                                </tr>
                            </table>
                            <p class="description"><?php esc_html_e( 'Leave empty to use the base cutoff. Use "closed" or "-" to disable same-day shipping on that day.', 'wcscc' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wcscc_holiday_dates"><?php esc_html_e( 'Holiday / blackout dates', 'wcscc' ); ?></label></th>
                        <td>
                            <textarea id="wcscc_holiday_dates" name="wcscc_holiday_dates" rows="3" cols="60"><?php echo esc_textarea( $settings['holiday_dates'] ); ?></textarea>
                            <p class="description"><?php esc_html_e( 'One date per line, format YYYY-MM-DD. On these dates, same-day shipping is disabled.', 'wcscc' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Country-specific rules', 'wcscc' ); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wcscc_enable_country_rules" value="1" <?php checked( $settings['enable_country_rules'], 1 ); ?> />
                                <?php esc_html_e( 'Use different cutoff times by shipping country.', 'wcscc' ); ?>
                            </label>
                            <?php if ( ! self::is_pro() ) : ?>
                                <p class="description"><em><?php esc_html_e( 'Requires an active Pro license.', 'wcscc' ); ?></em></p>
                            <?php endif; ?>
                            <textarea id="wcscc_country_rules" name="wcscc_country_rules" rows="4" cols="60"><?php echo esc_textarea( $settings['country_rules'] ); ?></textarea>
                            <p class="description"><?php esc_html_e( 'One per line, format: CC=HH:MM (e.g. US=14:00, GB=16:00). Shipping country is used if available, otherwise billing.', 'wcscc' ); ?></p>
                        </td>
                    </tr>
                </table>

                <hr/>

                <h2><?php esc_html_e( 'Pro – Badges & Labels', 'wcscc' ); ?></h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Enable badges', 'wcscc' ); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="wcscc_enable_badges" value="1" <?php checked( $settings['enable_badges'], 1 ); ?> />
                                <?php esc_html_e( 'Show a compact badge like "Ships today" next to products.', 'wcscc' ); ?>
                            </label>
                            <?php if ( ! self::is_pro() ) : ?>
                                <p class="description"><em><?php esc_html_e( 'Requires an active Pro license.', 'wcscc' ); ?></em></p>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wcscc_badge_today"><?php esc_html_e( 'Badge – today', 'wcscc' ); ?></label></th>
                        <td>
                            <input type="text" id="wcscc_badge_today" name="wcscc_badge_today" value="<?php echo esc_attr( $settings['badge_today'] ); ?>" class="regular-text" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wcscc_badge_tomorrow"><?php esc_html_e( 'Badge – tomorrow', 'wcscc' ); ?></label></th>
                        <td>
                            <input type="text" id="wcscc_badge_tomorrow" name="wcscc_badge_tomorrow" value="<?php echo esc_attr( $settings['badge_tomorrow'] ); ?>" class="regular-text" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wcscc_badge_later"><?php esc_html_e( 'Badge – later', 'wcscc' ); ?></label></th>
                        <td>
                            <input type="text" id="wcscc_badge_later" name="wcscc_badge_later" value="<?php echo esc_attr( $settings['badge_later'] ); ?>" class="regular-text" />
                            <p class="description"><?php esc_html_e( 'Use {date} for the estimated ship date.', 'wcscc' ); ?></p>
                        </td>
                    </tr>
                </table>

                <hr/>

                <h2><?php esc_html_e( 'License', 'wcscc' ); ?></h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Status', 'wcscc' ); ?></th>
                        <td>
                            <?php
                            if ( $status === 'valid' ) {
                                echo '<span style="color:green;font-weight:bold;">' . esc_html__( 'Active', 'wcscc' ) . '</span>';
                            } elseif ( $status === 'grace' ) {
                                echo '<span style="color:#e6a700;font-weight:bold;">' . esc_html__( 'Grace period', 'wcscc' ) . '</span>';
                            } else {
                                echo '<span style="color:#cc0000;font-weight:bold;">' . esc_html__( 'Inactive', 'wcscc' ) . '</span>';
                            }
                            ?>
                            <p class="description">
                                <?php
                                printf(
                                    esc_html__( 'License bound to domain: %s', 'wcscc' ),
                                    '<code>' . esc_html( $domain ) . '</code>'
                                );
                                ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wcscc_license_key"><?php esc_html_e( 'License key', 'wcscc' ); ?></label></th>
                        <td>
                            <input type="text" id="wcscc_license_key" name="wcscc_license_key" value="<?php echo esc_attr( $opt['key'] ); ?>" class="regular-text" />
                            <p class="description"><?php esc_html_e( 'Paste the license key from your account / purchase email.', 'wcscc' ); ?></p>
                        </td>
                    </tr>
                </table>

                <?php submit_button( __( 'Save changes', 'wcscc' ) ); ?>
            </form>
        </div>
        <?php
    }

    private function remote_request( $key, $action ) {
        $body = [
            'action'      => $action,
            'license_key' => $key,
            'product'     => WCSCC_PRODUCT_SLUG,
            'domain'      => $this->domain(),
        ];

        $response = wp_remote_post(
            WCSCC_LICENSE_API,
            [
                'timeout' => 10,
                'body'    => $body,
            ]
        );

        if ( is_wp_error( $response ) ) {
            return [
                'ok'     => false,
                'msg'    => $response->get_error_message(),
                'active' => false,
            ];
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );

        if ( 200 !== $code ) {
            return [
                'ok'     => false,
                'msg'    => 'HTTP ' . $code,
                'active' => false,
            ];
        }

        $data = json_decode( $body, true );
        if ( ! is_array( $data ) ) {
            return [
                'ok'     => false,
                'msg'    => 'Invalid JSON from license server',
                'active' => false,
            ];
        }

        $data = wp_parse_args(
            $data,
            [
                'ok'     => false,
                'msg'    => '',
                'active' => true,
            ]
        );

        return $data;
    }

    private function handle_validation_result( $res, $opt, $show_admin_notice = false, $new_key = null ) {
        $now = time();

        if ( null !== $new_key ) {
            $opt['key'] = $new_key;
        }

        if ( ! empty( $res['ok'] ) && ! empty( $res['active'] ) ) {
            $opt['activated']   = true;
            $opt['grace_until'] = 0;
            $opt['last_check']  = $now;

            update_option( self::OPT_KEY, $opt );
            set_transient( self::TRANSIENT_STATE, 'valid', self::CHECK_INTERVAL );

            if ( $show_admin_notice ) {
                $msg = $res['msg'] ? $res['msg'] : __( 'License validated.', 'wcscc' );
                add_action(
                    'admin_notices',
                    static function () use ( $msg ) {
                        echo '<div class="notice notice-success"><p>' . esc_html( $msg ) . '</p></div>';
                    }
                );
            }
            return;
        }

        // failed
        $opt['activated']  = false;
        $opt['last_check'] = $now;

        if ( empty( $opt['grace_until'] ) || $opt['grace_until'] < $now ) {
            $opt['grace_until'] = $now + ( self::GRACE_HOURS * HOUR_IN_SECONDS );
        }

        update_option( self::OPT_KEY, $opt );

        if ( $opt['grace_until'] > $now ) {
            set_transient( self::TRANSIENT_STATE, 'grace', self::CHECK_INTERVAL );
        } else {
            set_transient( self::TRANSIENT_STATE, 'invalid', self::CHECK_INTERVAL );
        }

        if ( $show_admin_notice ) {
            $msg = ! empty( $res['msg'] ) ? $res['msg'] : __( 'License validation failed.', 'wcscc' );
            add_action(
                'admin_notices',
                static function () use ( $msg ) {
                    echo '<div class="notice notice-warning"><p>' . esc_html( $msg ) . '</p></div>';
                }
            );
        }
    }

    private function domain() {
        $home = home_url();
        $host = wp_parse_url( $home, PHP_URL_HOST );
        return $host ? $host : $home;
    }
}

/**
 * Main Shipping Cutoff logic (frontend + product meta)
 */
class WCSCC_Plugin {

    /**
     * Prevent registering frontend hooks multiple times in the same request.
     *
     * Some checkout flows (especially with AJAX fragments / custom themes)
     * can cause the registration callback to run more than once, which
     * results in duplicated output.
     *
     * @var bool
     */
    private $frontend_hooks_registered = false;

    public function init() {
        if ( ! class_exists( 'WooCommerce' ) ) {
            return;
        }

        // Product meta (per-product Pro overrides).
        add_action( 'add_meta_boxes', [ $this, 'add_product_metabox' ] );
        add_action( 'save_post_product', [ $this, 'save_product_meta' ], 10, 2 );

        // Frontend display hooks.
        add_action( 'wp', [ $this, 'register_frontend_hooks' ] );

        // Shortcode.
        add_shortcode( 'wc_shipping_cutoff', [ $this, 'shortcode_handler' ] );
    }

    public function add_product_metabox() {
        add_meta_box(
            'wcscc_product_shipping',
            __( 'Shipping Cutoff (Pro)', 'wcscc' ),
            [ $this, 'render_product_metabox' ],
            'product',
            'side',
            'default'
        );
    }

    public function render_product_metabox( $post ) {
        wp_nonce_field( 'wcscc_save_product_meta', 'wcscc_meta_nonce' );

        $no_same_day    = get_post_meta( $post->ID, '_wcscc_no_same_day', true );
        $custom_cutoff  = get_post_meta( $post->ID, '_wcscc_custom_cutoff', true );

        if ( ! WCSCC_License::is_pro() ) {
            echo '<p><em>' . esc_html__( 'Per-product shipping rules are available with a Pro license.', 'wcscc' ) . '</em></p>';
            echo '<p>' . esc_html__( 'This product will follow the global shipping cutoff schedule from the plugin settings.', 'wcscc' ) . '</p>';
            return;
        }
        ?>
        <p>
            <label>
                <input type="checkbox" name="wcscc_no_same_day" value="1" <?php checked( $no_same_day, '1' ); ?> />
                <?php esc_html_e( 'Never ship same day (treat as next business day even before cutoff).', 'wcscc' ); ?>
            </label>
        </p>
        <p>
            <label for="wcscc_custom_cutoff"><?php esc_html_e( 'Custom cutoff time', 'wcscc' ); ?></label><br/>
            <input type="text" id="wcscc_custom_cutoff" name="wcscc_custom_cutoff" value="<?php echo esc_attr( $custom_cutoff ); ?>" placeholder="e.g. 13:00" style="width:100%;" />
            <span class="description"><?php esc_html_e( 'Optional. Leave empty to use global or weekday/country rules.', 'wcscc' ); ?></span>
        </p>
        <?php
    }

    public function save_product_meta( $post_id, $post ) {
        if ( ! isset( $_POST['wcscc_meta_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['wcscc_meta_nonce'] ) ), 'wcscc_save_product_meta' ) ) {
            return;
        }

        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        if ( 'product' !== $post->post_type ) {
            return;
        }

        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        if ( WCSCC_License::is_pro() ) {
            $no_same_day   = ! empty( $_POST['wcscc_no_same_day'] ) ? '1' : '';
            $custom_cutoff = isset( $_POST['wcscc_custom_cutoff'] ) ? sanitize_text_field( wp_unslash( $_POST['wcscc_custom_cutoff'] ) ) : '';

            if ( $no_same_day ) {
                update_post_meta( $post_id, '_wcscc_no_same_day', $no_same_day );
            } else {
                delete_post_meta( $post_id, '_wcscc_no_same_day' );
            }

            if ( $custom_cutoff !== '' ) {
                update_post_meta( $post_id, '_wcscc_custom_cutoff', $custom_cutoff );
            } else {
                delete_post_meta( $post_id, '_wcscc_custom_cutoff' );
            }
        }
    }

    public function register_frontend_hooks() {
        // Guard against duplicate hook registration.
        if ( $this->frontend_hooks_registered ) {
            return;
        }
        $this->frontend_hooks_registered = true;

        $settings = $this->get_settings();

        if ( ! WC()->cart ) {
            return;
        }

        if ( ! empty( $settings['enable_on_product'] ) ) {
            if ( ! has_action( 'woocommerce_single_product_summary', [ $this, 'output_on_single' ] ) ) {
                add_action( 'woocommerce_single_product_summary', [ $this, 'output_on_single' ], 25 );
            }
        }
        if ( ! empty( $settings['enable_on_cart'] ) ) {
            if ( ! has_action( 'woocommerce_cart_totals_after_order_total', [ $this, 'output_on_cart' ] ) ) {
                add_action( 'woocommerce_cart_totals_after_order_total', [ $this, 'output_on_cart' ] );
            }
        }
        if ( ! empty( $settings['enable_on_checkout'] ) ) {
            if ( ! has_action( 'woocommerce_review_order_after_order_total', [ $this, 'output_on_checkout' ] ) ) {
                add_action( 'woocommerce_review_order_after_order_total', [ $this, 'output_on_checkout' ] );
            }
        }
    }

    private function get_settings() {
        $defaults = [
            'base_cutoff_time'     => '16:00',
            'enable_on_product'    => 1,
            'enable_on_cart'       => 0,
            'enable_on_checkout'   => 0,
            'message_template'     => __( 'Order within {countdown} for dispatch today.', 'wcscc' ),
            'fallback_template'    => __( 'Order now for next business day dispatch.', 'wcscc' ),
            'enable_weekday_rules' => 0,
            'cutoff_monday'        => '',
            'cutoff_tuesday'       => '',
            'cutoff_wednesday'     => '',
            'cutoff_thursday'      => '',
            'cutoff_friday'        => '',
            'cutoff_saturday'      => '',
            'cutoff_sunday'        => '',
            'holiday_dates'        => '',
            'enable_badges'        => 0,
            'badge_today'          => __( 'Ships today', 'wcscc' ),
            'badge_tomorrow'       => __( 'Ships tomorrow', 'wcscc' ),
            'badge_later'          => __( 'Ships on {date}', 'wcscc' ),
            'enable_country_rules' => 0,
            'country_rules'        => '',
        ];
        $settings = get_option( 'wcscc_settings', [] );
        return wp_parse_args( $settings, $defaults );
    }

    /**
     * Shortcode handler: [wc_shipping_cutoff context="product|cart|checkout"]
     */
    public function shortcode_handler( $atts ) {
        $atts = shortcode_atts(
            [
                'context' => 'product',
            ],
            $atts,
            'wc_shipping_cutoff'
        );

        return $this->build_output( $atts['context'] );
    }

    public function output_on_single() {
        echo $this->build_output( 'product' );
    }

    public function output_on_cart() {
        echo $this->build_output( 'cart' );
    }

    public function output_on_checkout() {
        echo $this->build_output( 'checkout' );
    }

    /**
     * Build the countdown + badge HTML.
     *
     * @param string $context product|cart|checkout|shortcode
     * @return string
     */
    private function build_output( $context ) {
        $settings = $this->get_settings();
        $now      = $this->now();

        $holiday_dates = $this->parse_holidays( $settings['holiday_dates'] );
        $is_holiday    = in_array( $now->format( 'Y-m-d' ), $holiday_dates, true );

        $cutoff_dt = $this->determine_cutoff_datetime( $now, $settings );
        $can_same_day = $cutoff_dt && $now < $cutoff_dt && ! $is_holiday;

        // Per-product override: if current product disallows same-day, treat as next-day.
        $product_no_same_day = false;
        $current_product     = $this->get_context_product( $context );
        if ( $current_product && WCSCC_License::is_pro() ) {
            $no_same_day_meta = get_post_meta( $current_product->get_id(), '_wcscc_no_same_day', true );
            if ( $no_same_day_meta ) {
                $product_no_same_day = true;
            }
        }

        if ( $product_no_same_day ) {
            $can_same_day = false;
        }

        $html = '<div class="wcscc-wrapper" style="margin:8px 0;padding:8px 10px;border-radius:4px;background:#f5fbff;border:1px solid #c5e5ff;font-size:0.9em;">';

        if ( $can_same_day ) {
            $diff      = $cutoff_dt->getTimestamp() - $now->getTimestamp();
            $countdown = $this->format_diff( $diff );
            $time_str  = $cutoff_dt->format( get_option( 'time_format', 'H:i' ) );
            $date_str  = $cutoff_dt->format( get_option( 'date_format', 'Y-m-d' ) );

            $message = strtr(
                $settings['message_template'],
                [
                    '{countdown}' => $countdown,
                    '{time}'      => $time_str,
                    '{date}'      => $date_str,
                ]
            );

            $html .= '<div class="wcscc-message" style="color:#004b7a;">' . wp_kses_post( $message ) . '</div>';
            $badge_html = $this->build_badge( 'today', $cutoff_dt, $settings );
        } else {
            $next_business_day = $this->calculate_next_business_day( $now, $holiday_dates );
            $date_str          = $next_business_day->format( get_option( 'date_format', 'Y-m-d' ) );

            $message = strtr(
                $settings['fallback_template'],
                [
                    '{date}' => $date_str,
                ]
            );

            $html .= '<div class="wcscc-message" style="color:#7a4b00;">' . wp_kses_post( $message ) . '</div>';

            // Determine if the next business day is tomorrow or later.
            $tomorrow = (clone $now)->modify( '+1 day' )->format( 'Y-m-d' );
            $badge_type = ( $next_business_day->format( 'Y-m-d' ) === $tomorrow ) ? 'tomorrow' : 'later';
            $badge_html = $this->build_badge( $badge_type, $next_business_day, $settings );
        }

        if ( $badge_html ) {
            $html .= $badge_html;
        }

        $html .= '</div>';

        return $html;
    }

    /**
     * Get current product context if applicable.
     *
     * @param string $context
     * @return WC_Product|null
     */
    private function get_context_product( $context ) {
        if ( 'product' === $context && function_exists( 'wc_get_product' ) ) {
            global $product;
            if ( $product instanceof WC_Product ) {
                return $product;
            }
        }
        return null;
    }

    private function now() {
        $timezone = wp_timezone();
        return new DateTime( 'now', $timezone );
    }

    private function parse_holidays( $raw ) {
        $lines = array_filter(
            array_map(
                'trim',
                explode( "\n", (string) $raw )
            )
        );
        $dates = [];
        foreach ( $lines as $line ) {
            if ( preg_match( '/^\d{4}-\d{2}-\d{2}$/', $line ) ) {
                $dates[] = $line;
            }
        }
        return $dates;
    }

    /**
     * Determine cutoff DateTime for today based on settings (and Pro rules).
     *
     * @param DateTime $now
     * @param array    $settings
     * @return DateTime|null
     */
    private function determine_cutoff_datetime( DateTime $now, $settings ) {
        $cutoff_time = $settings['base_cutoff_time'];

        // Country-specific (Pro).
        if ( WCSCC_License::is_pro() && ! empty( $settings['enable_country_rules'] ) && ! empty( $settings['country_rules'] ) ) {
            $country_cutoff = $this->get_country_cutoff_time( $settings['country_rules'] );
            if ( $country_cutoff ) {
                $cutoff_time = $country_cutoff;
            }
        }

        // Weekday rules (Pro).
        if ( WCSCC_License::is_pro() && ! empty( $settings['enable_weekday_rules'] ) ) {
            $weekday_index = (int) $now->format( 'N' ); // 1=Mon..7=Sun
            $map           = [
                1 => 'cutoff_monday',
                2 => 'cutoff_tuesday',
                3 => 'cutoff_wednesday',
                4 => 'cutoff_thursday',
                5 => 'cutoff_friday',
                6 => 'cutoff_saturday',
                7 => 'cutoff_sunday',
            ];
            if ( isset( $map[ $weekday_index ] ) && ! empty( $settings[ $map[ $weekday_index ] ] ) ) {
                $val = trim( $settings[ $map[ $weekday_index ] ] );
                if ( in_array( strtolower( $val ), [ 'closed', '-', 'off' ], true ) ) {
                    return null; // no same-day shipping today
                }
                $cutoff_time = $val;
            }
        }

        // Per-product custom cutoff (Pro).
        $current_product = $this->get_context_product( 'product' );
        if ( $current_product && WCSCC_License::is_pro() ) {
            $custom = get_post_meta( $current_product->get_id(), '_wcscc_custom_cutoff', true );
            if ( $custom ) {
                $cutoff_time = $custom;
            }
        }

        if ( ! preg_match( '/^\d{1,2}:\d{2}$/', $cutoff_time ) ) {
            return null;
        }

        try {
            $cutoff = new DateTime( $now->format( 'Y-m-d' ) . ' ' . $cutoff_time, $now->getTimezone() );
            return $cutoff;
        } catch ( Exception $e ) {
            return null;
        }
    }

    /**
     * Parse country rules and return cutoff for current customer's country.
     *
     * @param string $raw
     * @return string|null HH:MM
     */
    private function get_country_cutoff_time( $raw ) {
        if ( ! WC()->customer ) {
            return null;
        }

        $country = WC()->customer->get_shipping_country();
        if ( ! $country ) {
            $country = WC()->customer->get_billing_country();
        }
        if ( ! $country ) {
            return null;
        }

        $lines = array_filter(
            array_map(
                'trim',
                explode( "\n", (string) $raw )
            )
        );
        foreach ( $lines as $line ) {
            if ( strpos( $line, '=' ) === false ) {
                continue;
            }
            list( $cc, $time ) = array_map( 'trim', explode( '=', $line, 2 ) );
            if ( strtoupper( $cc ) === strtoupper( $country ) && preg_match( '/^\d{1,2}:\d{2}$/', $time ) ) {
                return $time;
            }
        }
        return null;
    }

    /**
     * Calculate next business day (skipping weekends & holidays).
     *
     * @param DateTime $from
     * @param array    $holiday_dates
     * @return DateTime
     */
    private function calculate_next_business_day( DateTime $from, array $holiday_dates ) {
        $dt = clone $from;
        $dt->modify( '+1 day' );
        while ( true ) {
            $weekday = (int) $dt->format( 'N' ); // 1=Mon..7=Sun
            $date    = $dt->format( 'Y-m-d' );
            if ( $weekday >= 6 || in_array( $date, $holiday_dates, true ) ) {
                $dt->modify( '+1 day' );
                continue;
            }
            break;
        }
        return $dt;
    }

    private function format_diff( $seconds ) {
        $seconds = max( 0, (int) $seconds );
        $hours   = floor( $seconds / 3600 );
        $minutes = floor( ( $seconds % 3600 ) / 60 );

        if ( $hours > 0 ) {
            return sprintf( _n( '%d hour %d minutes', '%d hours %d minutes', $hours, 'wcscc' ), $hours, $minutes );
        }
        return sprintf( _n( '%d minute', '%d minutes', $minutes, 'wcscc' ), $minutes );
    }

    private function build_badge( $type, DateTime $date, $settings ) {
        if ( ! WCSCC_License::is_pro() || empty( $settings['enable_badges'] ) ) {
            return '';
        }

        $badge_text = '';
        if ( 'today' === $type ) {
            $badge_text = $settings['badge_today'];
        } elseif ( 'tomorrow' === $type ) {
            $badge_text = $settings['badge_tomorrow'];
        } else {
            $badge_text = strtr(
                $settings['badge_later'],
                [
                    '{date}' => $date->format( get_option( 'date_format', 'Y-m-d' ) ),
                ]
            );
        }

        if ( '' === trim( $badge_text ) ) {
            return '';
        }

        return '<div class="wcscc-badge" style="display:inline-block;margin-top:6px;padding:2px 6px;border-radius:999px;background:#e5f9e7;color:#236b2e;font-size:0.8em;">' . esc_html( $badge_text ) . '</div>';
    }
}

// Bootstrap license & plugin.
add_action(
    'plugins_loaded',
    static function () {
        if ( ! class_exists( 'WooCommerce' ) ) {
            return;
        }

        $lic = new WCSCC_License();
        $lic->init();

        $core = new WCSCC_Plugin();
        $core->init();
    }
);
